# coding: utf-8

from lxml import html

from at.common import exceptions
from at.aux_ import Mailing
from at.common import ServiceAccess
from at.common.Context import ContextBinder
from at.common import BodyFormatter
from at.common import utils
from at.common import const
from at.common.sender import sender_client
from at.pump.Query import Handler, Event
from at.pump import Digest
from at.pump.Mailer import *

_log = logging.getLogger(__name__)


class MailCantBeSent(Exception):
    pass


class MailHandler(Handler):
    setting_name = None
    notification_class = Notification

    def get_additional_context(self, notification):
        return {
            'language': notification.target.language
        }

    def check_auth(self, uid):
        return ServiceAccess.check_accessibility(uid)

    def enabled(self, notification):
        if notification.target.type == 'mail':
            return True
        mail_settings = Mailing.get_user_mail_settings(notification.target.uid)
        return (
            self.setting_name is None or
            getattr(mail_settings, self.setting_name)
        ) and not notification.target.dismissed

    def action(self, event, debug=False):
        notification = self.get_notification(event)
        try:
            assert notification.target.type != 'club', 'Logic error: Notifications should not be ever sent to a club.'
        except exceptions.NotFound:
            # No such user
            return event.get_human_repr() + ' no such user'
        if not debug and not self.enabled(notification):
            return True
        try:
            sent = self.send(notification)
            return event.get_human_repr() + ' ' + str(sent)
        except exceptions.NotFound:
            _log.warning('Post or comment seems to be deleted')
            return event.get_human_repr() + ' deleted'

    def get_notification(self, event):
        return self.notification_class(*event)

    def get_fromto(self, who):
        return utils.force_unicode(who.email)

    def get_data(self, notification):
        body = notification.render()
        subject = html.fromstring(body).find('head/title').text
        return {
            "sender": '%s (%s)' % (
                utils.force_unicode(notification.initiator.title),
                utils.force_unicode(notification.initiator.login),
            ),
            "replyto": self.get_fromto(notification.initiator),
            # каждому надо посылать отдельно, т.к. язык у каждого свой
            # и так как в каждом письме свои отношения между отправителем,
            # получателем и прочими действующими лицами.
            "receiver": self.get_fromto(notification.target),
            "subject": subject,
            "body": html.tostring(html.fromstring(body).find('body/'), encoding='unicode'),
            "headers": notification.get_headers(),
        }

    def send(self, notification):
        if settings.SKIP_EMAIL_HANDLING:
            _log.info('Skip email sending')
            return

        is_feed = isinstance(notification.target, BodyFormatter.FeedRepresentation)
        if is_feed and not self.check_auth(notification.target.uid):
            _log.warning(
                "Notifications disabled for non-yandex employees %s",
                notification.target.uid
            )
            return False
        with ContextBinder(**self.get_additional_context(notification)):
            data = self.get_data(notification)
            log_message = "Email from: {} to: {} | Notification type: {}".format(
                data['sender'],
                data['receiver'],
                notification.__class__.__name__,
            )
            if isinstance(notification, PostNotification):
                post_log_message = " | Post_id {}.{}.{}".format(
                    notification.feed_id,
                    notification.item_no,
                    notification.comment_id
                )
                log_message += post_log_message
            try:
                sent = send_email(
                    subject=data['subject'],
                    body=data['body'],
                    sender=data['sender'],
                    receiver=data['receiver'],
                    headers=data.get('headers') and dict(data['headers']),
                    msgtype=notification.msgtype,
                )
                _log.info(log_message)
            except Exception:
                _log.exception(log_message)
                raise
            return sent


def send_email(subject, body, sender, receiver, headers, msgtype):
    if headers and 'References' in headers and 'In-Reply-To' in headers:
        headers['References'] += (" " + headers['In-Reply-To'])

    try:
        return sender_client.send_message(
            subject=subject,
            sender=sender,
            receiver=receiver,
            headers=headers,
            campaign_slug=settings.MAILING_TEMPLATES[msgtype],
            template_args={'content': body},
        )

    except Exception as exc:
        log.error(f"Message sending to {receiver} failed:\n{exc}")


class MailNewRelationHandler(MailHandler):
    notification_class = InviteNotification
    setting_name = 'notify_invites'

    class Event(Event):
        field_names = ['target', 'initiator_id', 'feed_id', 'relation_type']

    def action(self, event, debug=False):
        return super(MailNewRelationHandler, self).action(event, debug)


class MailEntryHandler(MailHandler):
    def get_additional_context(self, notification):
        # Для уведомлений о каком-либо элементе прокидываю идентификаторы в контекст
        # используются для формирования ссылок в xsl шаблоне заглушки о медиа-контенте
        return {
            'language': notification.target.language,
            'feed_id': notification.feed_id,
            'item_no': notification.item_no,
            'comment_id': notification.comment_id
        }


class MailMentionHandler(MailEntryHandler):
    notification_class = MentionNotification

    class Event(Event):
        field_names = ['target', 'initiator_id',
                       'feed_id', 'item_no', 'comment_id', 'mention_type']


class MailNewPostHandler(MailEntryHandler):
    notification_class = PostNotification

    class Event(Event):
        field_names = ['target', 'initiator_id', 'feed_id', 'item_no']


class MailNewCommentHandler(MailEntryHandler):
    notification_class = CommentNotification
    setting_name = None

    class Event(Event):
        field_names = ['target', 'initiator_id', 'feed_id', 'item_no',
                       'comment_id']


class MailPremoderatedPostHandler(MailEntryHandler):
    notification_class = ModerationNotification
    setting_name = 'notify_moderation'

    class Event(Event):
        field_names = ['target', 'initiator_id', 'feed_id', 'item_no']


class MailDigestHandler(MailHandler):
    notification_class = DigestNotification
    setting_name = None

    class Event(Event):
        field_names = ['target', 'mode']

    def action(self, event, debug=False):
        target, mode = event
        assert mode in (
            const.DIGEST_MODES.FULL,
            const.DIGEST_MODES.POPULAR,
        ), "Bad digest mode!"
        popular_mode = mode == const.DIGEST_MODES.POPULAR
        post_set = Digest.DigestSet.get_digest_set(target, popular_mode)
        if not post_set:
            _log.debug('No post in digest for user %s, skip', target)
            return True
        sent = self.send(self.get_notification((target, post_set)))
        return '.'.join(map(str, [target, mode, sent]))


import types

names = list(locals().keys())
__all__ = [name for name, obj in locals().items() if isinstance(obj, type) and issubclass(obj,
                                                 (Event, Handler))]
