# coding: utf-8

import datetime
import logging

from xml.sax.saxutils import escape
from mako.lookup import TemplateLookup

from django.conf import settings

from at.common import KeyRing
from at.common import decorators
from at.common.BodyFormatter import (
    get_firstline,
    fix_text,
    makeRepresentation,
    remove_tags,
)
from at.common.Context import context
from at.aux_ import entries


log = logging.getLogger(__name__)


class HeadersBuilder(object):

    # XXX FIXME выкинуть эту помойку к чёрту
    def __call__(self, notification_type, headers):
        b_headers = [('X-Yaru-Notification-Type', notification_type)]
        for name, args in list(headers.items()):
            b_headers.append(self.header(name, args))
        return b_headers

    def header(self, name, args):
        header, value =  getattr(self, '_build_%s' % name)(*args)
        return (header, value)

    def _build_message_id(self, target_id, feed_id, item_no, comment_id):
        return ('Message-Id',
                '<%s@my.yandex-team.ru>' %\
                    KeyRing.sign_message(target_id,
                                     'user.uid.%d.%d.%d' %\
                                             (feed_id, item_no, comment_id or 0)))

    def _build_in_reply_to(self, target_id, feed_id, item_no, parent_id):
        return ('In-Reply-To',
                '<%s@my.yandex-team.ru>' %\
                     KeyRing.sign_message(target_id,
                                     'user.uid.%d.%d.%d' %\
                                     (feed_id, item_no, parent_id or 0)))

    def _build_references(self, target_id, feed_id, item_no):
        return ('References',
                '<%s@my.yandex-team.ru>' %\
                    KeyRing.sign_message(target_id,
                                     'user.uid.%d.%d_post' %\
                                             (feed_id, item_no)))
    def _build_feed(self, feed_id):
        return ('X-Atushka-Feed', str(feed_id))

    def _build_post(self, item_no):
        return ('X-Atushka-Post', str(item_no))


build_headers_for = HeadersBuilder()

TPL_LOOKUP_KW_ENC = {'input_encoding': 'utf-8', 'output_encoding': 'utf-8'}


class Notification(object):

    msgtype = None
    templates = None

    def __init__(self, target, sender):
        self._headers_info = {}
        self.initiator = makeRepresentation(sender)
        self.target = makeRepresentation(target)
        self.subject = NotImplemented
        # Additional context variables
        self.datetime = datetime
        self.fields = ['initiator', 'target', 'datetime']

    @property
    def template(self):
        if self.templates is None:
            lookup = TemplateLookup(
                directories=[settings.MAKO_PATH], **TPL_LOOKUP_KW_ENC)

            get_filename = lambda lang: '%s/%s.mako.html' % (lang, self.msgtype)
            self.templates = dict(
                (
                    lang,
                    lookup.get_template(get_filename(lang))
                )
                for lang in ['ru', 'en']
            )
        return self.templates[context.language]

    def get_headers(self):
        return build_headers_for(self.msgtype, self._headers_info)

    def render(self):
        return self.template.render_unicode(**dict((k, getattr(self, k)) for k in self.fields))


class PostWrapper(object):
    # XXX FIXME: порефакторить, чтоб parent стал объектом, а не префиксом имён атрибутов
    def __init__(self, feed_id, item_no, comment_id = 0):
        self.__tags = None
        self.__firstline = None
        self.feed_id = feed_id
        self.item_no = item_no
        self.comment_id = comment_id
        self.post_type = None
        self.reply_type = None
        self.parent_id = 0
        self.parent_body = ''
        self.parent_type = None
        self.feed = makeRepresentation(self.feed_id)
        self.url = self.feed.get_url(self.item_no, self.comment_id)

        self.post = post = entries.models.load_entry(feed_id, item_no, 0)
        self.comment = comment = entries.models.load_entry(feed_id, item_no, comment_id)
        self.raw_body = post.body
        self.score = post.score
        self.xml_body = fix_text(self.raw_body)

        # костыль для того, чтобы в имейл попадал валидный html без
        # самозакрытых div
        from lxml import html
        try:
            self.body = html.tostring(html.fromstring(self.xml_body), encoding='unicode')
        except Exception:
            log.error('Failed xml -> html convertion. %s', self.xml_body)
            self.body = self.xml_body
        self.post_author = makeRepresentation(post.author_id)
        self.post_type = post.entry_type
        self.post_date = post.store_time

        self.title = escape(post.title or '')
        self.firstline = escape(self.get_firstline())
        if comment_id:
            self.reply_body =  fix_text(comment.body)
            self.reply_type = comment.entry_type
            self.reply_author = makeRepresentation(comment.author_id)
        if comment.has_parent:
            self.parent_body = fix_text(comment.parent.body)
            self.parent_type = comment.parent.type
            parent_author_id = comment.parent.author_id
            self.parent_id = comment.parent.comment_id
            self.parent_date = comment.parent.store_time
        else: # какая-то странная логика
            self.parent_date = post.store_time
            parent_author_id = post.author_id
            self.parent_body = post.snippet

        self.parent_author = makeRepresentation(parent_author_id)
        self.parent_url = self.feed.get_url(self.item_no, self.parent_id)
        self.collect_additional_data()

    def collect_additional_data(self):
        makeRepresentationOrNone = decorators.or_none(makeRepresentation)
        post, comment = self.post, self.comment
        if post.type == 'congratulation':
            self.congratulated_by_post = makeRepresentationOrNone(post.whom)
        if comment.type == 'congratulation':
            self.congratulated_by_reply = makeRepresentationOrNone(comment.whom)
        if comment.type == 'summon':
            self.summoned = makeRepresentationOrNone(comment.summon_uid or comment.summon_address)
        if comment.has_parent and comment.parent.type == 'summon':
            self.summoned_in_parent = makeRepresentationOrNone(comment.parent.summon_uid or
                                                               comment.parent.summon_address)
        if post.type in ('friend', 'unfriend'):
            self.friended_in_post = post.friender
        if post.type in ('join', 'unjoin'):
            self.friended_in_post = post.club

        if comment.type in ('friend', 'unfriend'):
            self.friended_in_reply = comment.friender
        if comment.type in ('join', 'unjoin'):
            self.friended_in_reply = comment.club

        if comment.has_parent and comment.parent.type in ('friend', 'unfriend'):
            self.friended_in_reply = comment.parent.friender
        if comment.has_parent and comment.parent.type in ('join', 'unjoin'):
            self.friended_in_reply = comment.parent.club
        self.store_time = post.store_time

    @property
    def tags(self):
        if self.__tags is None:
            self.__tags = dict((tag.id, tag.title) for tag in self.post.tags)
        return self.__tags

    def get_firstline(self, maxlen=78):
        if self.__firstline is None:
            self.__firstline = get_firstline(
                    self.xml_body, maxlen)
        return self.__firstline

    def get_snippet(self, maxlen=1000):
        return self.post._make_snippet(maxlen)

    snippet = property(get_snippet)

    def is_empty(self, text):
        # Иногда будут ложные срабатывания,
        # если весь пост состоит из одного тега (<img src=".."/>)
        # но это лучше чем сейчас
        return not bool(remove_tags(text))

    def is_body_empty(self):
        return self.is_empty(self.xml_body)

    def is_reply_body_empty(self):
        return self.is_empty(self.reply_body)



class DigestNotification(Notification):
    msgtype = 'digest'

    def __init__(self, target, posts):
        super(DigestNotification, self).__init__(target, makeRepresentation(settings.ROBOT_UID))
        self.posts = posts
        self.fields += ['posts']


class PostNotification(Notification):
    msgtype = 'post'
    def __init__(self, target, sender, feed_id, item_no, comment_id = 0):
        super(PostNotification, self).__init__(target, sender)
        self.feed = makeRepresentation(feed_id)
        self.feed_id = feed_id
        self.item_no = item_no
        self.comment_id = comment_id
        self._post = None

        self.fields += ['feed', 'post']

    def get_headers(self):
        # FIXME как-то неизящно; и действительно ли они не нужны для рассылок?
        # Кроме того, кажется, references должен содержать весь список id в треде,
        # а не только самый первый.
        if self.target.type != 'mail':
            if self.comment_id:
                # Делать референсес только комментам
                self._headers_info['references'] = (
                    self.target.uid, self.feed.uid, self.post.item_no)
                self._headers_info['in_reply_to'] = (
                    self.target.uid, self.feed.uid, self.post.item_no, self.post.parent_id)
            self._headers_info['message_id'] = (
                    self.target.uid, self.feed.uid, self.post.item_no, self.post.comment_id)
        self._headers_info['feed'] = (
            self.feed.uid,
        )
        self._headers_info['post'] = (
            self.post.item_no,
        )
        return super(PostNotification, self).get_headers()

    @property
    def post(self):
        if self._post is None:
            self._post = PostWrapper(self.feed_id, self.item_no, self.comment_id)
        return self._post




class ModerationNotification(PostNotification):
    msgtype = 'moderate'

class CommentNotification(PostNotification):
    msgtype = 'reply'


class MentionNotification(PostNotification):
    msgtype = 'mention'
    def __init__(self, target, sender, feed_id, item_no, comment_id = 0, mention_type = 'USER'):
        super(MentionNotification, self).__init__(target, sender, feed_id, item_no, comment_id)
        self.mention_type = mention_type
        self.fields += ['mention_type']


class InviteNotification(Notification):
    msgtype = 'invite'
    def __init__(self, target, sender, feed, relation):
        super(InviteNotification, self).__init__(target, sender)
        # XXX NB: field "deleted" is ignored (events are skipped in Handler)
        self.feed = makeRepresentation(feed)
        self.relation = relation
        self.fields += ['feed', 'relation']

    def get_headers(self):
        self._headers_info['feed'] = (
            self.feed.uid,
        )
        return super(InviteNotification, self).get_headers()


