# coding: utf-8

import logging

from at.common import utils
from django.conf import settings
from at.common import postlink
from at.aux_ import Mailing
from at.aux_ import entries
from at.aux_ import Accesses
from at.aux_.UserEvents import EventList, MentionNotification
from at.pump.Query import Handler, Event
from at.pump.MailHandlers import MailNewCommentHandler, MailNewPostHandler
from at.pump import HandlerRegistry


_log = logging.getLogger(__name__)


class DeliverHandler(Handler):

    def get_entry_author(self, feed_id, item_no, comment_id=0):
        ai = utils.getGodAuthInfo()
        entry = entries.models.load_entry(feed_id, item_no, comment_id)
        return entry.author_id


# Ищет в новом или отредактированном комментарии упоминания других пользователей.
# Уведомляет других пользователей о новом упоминании.
class DeliverMentionsHandler(DeliverHandler):
    class Event(Event):
        field_names = ['feed_id', 'item_no', 'comment_id', 'timestamp']

    def action(self, event):
        feed_id, item_no, comment_id, timestamp = event

        # XXX поменять на что-будь более разумное
        author_uid = self.get_entry_author(feed_id, item_no, comment_id)
        if author_uid is None:
            msg = "Post or comment had been deleted"
            _log.warning(msg)
            return event.get_human_repr() + ' deleted'

        # Загружаем тело поста/комментария.
        entry = entries.load_entry(feed_id, item_no, comment_id)

        mentions = dict()
        body = entry.body or entry.body_original
        # Ищем ссылки других пользователей, на их дневники, посты и комментарии.
        for mentioned_uid, mention_type, mentioned_item_tuple in self.find_mentions(body):
            mentions[mentioned_uid] = (mention_type, mentioned_item_tuple)

        # Сохраняем уведомления о найденных упоминаниях.
        # Не отсылаем уведомление тем, кому придёт уведомление о комментарии.
        notified_uids = set([author_uid]) | Mailing.Mailing.GetMailingTargets(feed_id, item_no, comment_id)
        for uid, (mention_type, mentioned_entry) in mentions.items():
            if uid < 0 or uid >= settings.COMMUNITY_START_ID:
                _log.error("Incorrect mentioned uid: %r." % (uid,))
                continue
            _log.warning("Has %s/%r mention of %r." % (mention_type, mentioned_entry, uid))
            if uid in notified_uids:
                _log.warning("Do not deliver mention of uid = %d, because user is subscribed." % (uid,))
                continue

            if entry.is_comment:
                post = entries.load_entry(feed_id, item_no)
            else:
                post = entry
            if not Accesses.Access(uid, feed_id, post).can_read_post():
                _log.warning("Do not deliver mention of uid = %d, because post is restricted." % (uid,))
                continue

            if entry.has_trackback:
                feed_id, item_no, comment_id = entry.tb_feed_id, entry.tb_item_no, 0

            without_duplicate = EventList(uid).push(
                MentionNotification(
                    (feed_id, item_no, comment_id),
                    author_uid, mention_type, utils.parse_isoformat(timestamp),
                    mentioned_entry
                )
            )
            if without_duplicate:
                # Добавлено новое уведомление, хотим учесть его как ответ.
                HandlerRegistry.put_event(
                    'ModifyCommentsCounterHandler',
                    recipient_uid=uid,
                    delta=1,
                )
                # Высылаем почтовое уведомление.
                HandlerRegistry.put_event(
                    'MailMentionHandler',
                    target=uid,
                    initiator_id=author_uid,
                    feed_id=feed_id,
                    item_no=item_no,
                    comment_id=comment_id,
                    mention_type=mention_type,
                )
        return True

    def find_mentions(self, body):
        url_parser = postlink.MyUrlParser()
        for (feed_id, item_no, comment_id) in url_parser.finditer(body):
            if item_no:
                author_uid = self.get_entry_author(feed_id, item_no, comment_id)
                if author_uid:
                    yield  (author_uid,
                            'COMMENT' if comment_id else 'POST',
                            (feed_id, item_no, comment_id)
                           )
            else:
                if not utils.is_community_id(feed_id):
                    yield feed_id, 'USER', None


class DeliverCongratulationsHandler(DeliverHandler):
    class Event(Event):
        field_names = ['feed_id', 'item_no', 'comment_id', 'timestamp']

    def action(self, event):
        feed_id, item_no, comment_id, timestamp = event

        # XXX поменять на что-будь более разумное
        author_uid = self.get_entry_author(feed_id, item_no, comment_id)
        if author_uid is None:
            _log.error("Post/comment had been deleted.")
            return True

        god_ai = utils.getGodAuthInfo()
        entry = entries.load_entry(feed_id, item_no, comment_id)

        assert entry.type == 'congratulation', "Can't send congratulation from non-congratulation"

        whom = entry.whom
        if not whom:
            return

        if entry.has_trackback:
            feed_id, item_no, comment_id = entry.tb_feed_id, entry.tb_item_no, 0

        without_duplicate = EventList(whom).push(
            MentionNotification(
                (feed_id, item_no, comment_id),
                author_uid, 'CONGRATULATION', utils.parse_isoformat(timestamp),
                None
            )
        )
        if without_duplicate:
            if comment_id:
                HandlerRegistry.put_event(
                    'MailNewCommentHandler',
                    target=whom,
                    initiator_id=author_uid,
                    feed_id=feed_id,
                    item_no=item_no,
                    comment_id=comment_id,
                )
            else:
                HandlerRegistry.put_event(
                    'MailNewPostHandler',
                    target=whom,
                    initiator_id=author_uid,
                    feed_id=feed_id,
                    item_no=item_no,
                )


class ModifyCommentsCounterHandler(Handler):
    class Event(Event):
        field_names = ['recipient_uid', 'delta']

    def action(self, event):
        recipient_uid, delta = event
        with utils.get_connection() as connection:
            query = """INSERT INTO RecentActionsCounter(person_id, type, count, answered) VALUES(%s, %s, %s, %s)
                        ON DUPLICATE KEY UPDATE count = count + %s"""
            # В answered = 2 подсчитываем общее число упоминаний, это хак.
            # Эта таблица всё равно будет переделана под подсчёт общего числа полученных комментариев.
            connection.execute(query, (recipient_uid, 'comment_me', delta, 2, delta))
        return True


class CleanupMentionsHandler(Handler):
    class Event(Event):
        field_names = ['feed_id', 'item_no', 'comment_id']

    def action(self, event):
        feed_id, item_no, comment_id = event
        query = {
            'object.feed_id': feed_id,
            'object.item_no': item_no
        }
        if comment_id:
            query['object.comment_id'] = comment_id
        recipients = [ document['uid'] for document in MentionNotification.storage().find(query, {'uid': 1}) ]
        for uid in recipients:
            HandlerRegistry.put_event(
                'ModifyCommentsCounterHandler',
                recipient_uid=uid,
                delta=-1,
            )
            HandlerRegistry.put_event(
                'RemoveMentionNotificationsHandler',
                feed_id=feed_id,
                item_no=item_no,
                comment_id=comment_id,
            )
        return True


class RemoveMentionNotificationsHandler(Handler):
    class Event(Event):
        field_names = ['feed_id', 'item_no', 'comment_id']

    def action(self, event):
        feed_id, item_no, comment_id = event
        query = {
            'object.feed_id': feed_id,
            'object.item_no': item_no
        }
        if comment_id:
            query['object.comment_id'] = comment_id
        error = MentionNotification.storage().delete_one(query)
        return True


import types
names = list(locals().keys())
__all__ = [name for name, obj in locals().items() if isinstance(obj, type) and issubclass(obj,
                                                 (Event, Handler))]
