# coding: utf-8
import types
import logging
import datetime

from at.common import utils
from at.common.feed_weight_counter import FeedWeightCounter
from at.common import exceptions
from at.aux_ import Mailing
from at.aux_ import Community
from at.aux_ import entries
from at.pump import HandlerRegistry
from at.aux_ import models
from at.pump.Query import Handler, Event


logger = logging.getLogger(__name__)

LOGGER_REG_CALLBACK_MESSAGE = 'Register new callback | Post_id: {}.{}.{} | Callback {}'


class SummonFinalizeHandler(Handler):
    class Event(Event):
        field_names = ['initiator', 'feed_id', 'item_no', 'comment_id', 'target']

    def action(self, event):
        initiator, feed_id, item_no, comment_id, target = event
        try:
            target = int(target)
            email = False
        except ValueError:
            if '@' not in target:
                logger.warning('No valid summon meta in element %s.%s.%s', feed_id, item_no, comment_id)
                return True
            email = True

        known_targets = set([initiator]) | Mailing.Mailing.GetMailingTargets(feed_id, item_no, comment_id)
        if email:
            if target in [p.email for p in list(models.Person.get_persons(known_targets).values())]:
                return True
        else:
            if target in known_targets:
                return True

        if not email:
            HandlerRegistry.put_event(
                'InsertRecentHandler',
                target, feed_id, item_no, comment_id,
                datetime.datetime.now().isoformat())
            if not Mailing.get_user_mail_settings(target).notify_summons:
                # Don't send summon mail if disabled
                return True
        HandlerRegistry.put_event(
            'MailNewCommentHandler',
            target, initiator, feed_id, item_no, comment_id)
        return event.get_human_repr()


class CCFinalizeHandler(Handler):
    class Event(Event):
        field_names = ['feed_id', 'item_no', 'author_uid', 'cc_address']

    def action(self, event):
        feed_id, item_no, author_uid, cc_address = event
        # logger.debug("Scheduling CC: %s.%s -> %s" % (feed_id, item_no, cc_address))
        if not Mailing.is_email_normal(cc_address):
            logger.warning("Post_id: {}.{}.{} | Invalid email: {}".format(
                feed_id,
                item_no,
                '*',
                cc_address
            ))
            return
        known_target_emails = [
            p.email for p in list(models.Person.get_persons(
                Mailing.Mailing.GetMailingTargets(feed_id, item_no)
            ).values())
        ]
        if cc_address not in known_target_emails:
            logger.info(LOGGER_REG_CALLBACK_MESSAGE.format(
                feed_id,
                item_no,
                '*',
                'MailNewPostHandler'
            ) + "| CC addr: %s" % cc_address)
            HandlerRegistry.put_event(
                'MailNewPostHandler',
                cc_address, author_uid, feed_id, item_no)
        return event.get_human_repr() + ' ' + cc_address


class PremoderateHandler(Handler):
    accepts_new_style = True

    class Event(Event):
        field_names = ['person_id', 'feed_id', 'item_no']

    def action(self, event):
        moderators = Community.get_alive_moderators(event.fields['feed_id'])
        for moder_id in moderators:
            HandlerRegistry.put_event(
                'ClubNotifyHandler',
                feed_id=event.fields['feed_id'],
                action_type='moderation',
                item_no=event.fields['item_no'],
                squealer_id=event.fields['person_id'],
                moderator_id=moder_id,
            )
            HandlerRegistry.put_event(
                'MailPremoderatedPostHandler',
                target=moder_id,
                initiator_id=event.fields['person_id'],
                feed_id=event.fields['feed_id'],
                item_no=event.fields['item_no'],
            )


class ModerateHandler(Handler):
    accepts_new_style = True

    class Event(Event):
        field_names = ['feed_id', 'item_no']

    def action(self, event):
        if not utils.is_community_id(event.fields['feed_id']):
            return

        HandlerRegistry.put_event(
            'ClubCheckHandler',
            feed_id=event.fields['feed_id'],
            action_type='moderation',
            item_no=event.fields['item_no'],
            comment_id=0,
            moderator_id=0,

        )


class EditHandler(Handler):
    accepts_new_style = True

    class Event(Event):
        field_names = ['feed_id', 'item_no', 'comment_id']

    def action(self, event):
        """Выполнить действия, повторяющиеся на каждое редактирование:
        - скоринг
        - пуш в интрасерч
        - апдейт статистики клубов
        """
        logger_message = LOGGER_REG_CALLBACK_MESSAGE.format(
            event.fields['feed_id'],
            event.fields['item_no'],
            event.fields['comment_id'],
            '{}'
        )
        # скоринг делается не комменту, а корневому посту – и это правильно:
        logger.info(logger_message.format('ScorePostHandler'))
        HandlerRegistry.put_event(
            'ScorePostHandler',
            feed_id=event.fields['feed_id'],
            item_no=event.fields['item_no'],
        )
        # в поиск тоже пушится не отдельный коммент, а корневой пост (точнее, его id)
        logger.info(logger_message.format('ISearchHandler'))
        HandlerRegistry.put_event(
            'ISearchHandler',
            feed_id=event.fields['feed_id'],
            item_no=event.fields['item_no'],
            deleted=0,
        )
        if utils.is_community_id(event.fields['feed_id']):
            logger.info(logger_message.format('CommunityStatHandler'))
            HandlerRegistry.put_event(
                'CommunityStatHandler',
                feed_id=event.fields['feed_id'],
            )
        return event.get_human_repr()


class PublishHandler(Handler):
    accepts_new_style = True
    NO_NOTIFICATION_TYPES = ['friend',
                             'unfriend',
                             'join',
                             'unjoin']

    class Event(Event):
        field_names = ['person_id', 'feed_id', 'item_no', 'comment_id', 'store_time']

    def action(self, event):
        """Выполнить действия, происходящие при первой публикации:
        - отправка упоминаний
        - отправка поздравлений
        - отправка почтовых уведомлений"""
        try:
            initiator = utils.getAuthInfo(event.fields['person_id'], '_')
            entry = entries.load_entry(event.fields['feed_id'],
                                       event.fields['item_no'],
                                       event.fields['comment_id'])

        except exceptions.NotFound as e:
            logger.warning('PublishHandler: %s', e)
            return
        if utils.is_community_id(entry.feed_id):
            # TODO: Вынести в хендлер
            FeedWeightCounter(initiator.uid) \
                .increment(entry.feed_id)
        self.handle_common(event, entry)
        if entry.is_comment:
            self.handle_comment(event, entry)
        else:
            self.handle_post(event, entry)
        return event.get_human_repr()

    def handle_common(self, event, entry):
        now = datetime.datetime.now()
        HandlerRegistry.put_event(
            'DeliverMentionsHandler',
            feed_id=entry.feed_id,
            item_no=entry.item_no,
            comment_id=entry.comment_id,
            timestamp=now.isoformat(),
        )
        if entry.type == 'congratulation':
            HandlerRegistry.put_event(
                'DeliverCongratulationsHandler',
                feed_id=entry.feed_id,
                item_no=entry.item_no,
                comment_id=entry.comment_id,
                timestamp=now.isoformat(),
            )

    def handle_comment(self, event, _):
        HandlerRegistry.put_event(
            'RecentCommentsHandler',
            person_id=event.fields['person_id'],
            feed_id=event.fields['feed_id'],
            item_no=event.fields['item_no'],
            comment_id=event.fields['comment_id'],
            store_time=event.fields['store_time'],
        )
        comment_mail_notification_targets = Mailing.Mailing.GetMailingTargets(
            event.fields['feed_id'],
            event.fields['item_no'],
            event.fields['comment_id'],
        )

        for target in comment_mail_notification_targets:
            HandlerRegistry.put_event(
                'MailNewCommentHandler',
                target=target,
                initiator_id=event.fields['person_id'],
                feed_id=event.fields['feed_id'],
                item_no=event.fields['item_no'],
                comment_id=event.fields['comment_id'],
            )

    def handle_post(self, event, entry):
        if entry.type not in self.NO_NOTIFICATION_TYPES:
            post_notification_targets = Mailing.Mailing.GetMailingTargets(
                event.fields['feed_id'],
                event.fields['item_no'],
            )
            for target in post_notification_targets:
                HandlerRegistry.put_event(
                    'MailNewPostHandler',
                    target=target,
                    initiator_id=event.fields['person_id'],
                    feed_id=event.fields['feed_id'],
                    item_no=event.fields['item_no'],
                )


class DeleteHandler(Handler):
    accepts_new_style = True

    class Event(Event):
        field_names = ['person_id', 'feed_id', 'item_no', 'comment_id', 'on_moderation']

    def handle_common(self, event):
        comment_id = event.fields['comment_id'] or 0
        HandlerRegistry.put_event(
            'CleanupMentionsHandler',
            feed_id=event.fields['feed_id'],
            item_no=event.fields['item_no'],
            comment_id=comment_id,
        )
        HandlerRegistry.put_event(
            'CleanupLikesHandler',
            feed_id=event.fields['feed_id'],
            item_no=event.fields['item_no'],
            comment_id=comment_id,
        )
        return event.get_human_repr()

    def handle_post(self, event):
        on_moderation = event.fields['on_moderation']
        HandlerRegistry.put_event(
            'ISearchHandler',
            feed_id=event.fields['feed_id'],
            item_no=event.fields['item_no'],
            deleted=1,
        )

        if utils.is_community_id(event.fields['feed_id']) and on_moderation:
            HandlerRegistry.put_event(
                'ClubCheckHandler',
                feed_id=event.fields['feed_id'],
                action_type='moderation',
                item_no=event.fields['item_no'],
                comment_id=event.fields['comment_id'],
                moderator_id=0
            )

    def handle_comment(self, event):
        HandlerRegistry.put_event(
            'DeletedCommentHandler',
            person_id=event.fields['person_id'],
            feed_id=event.fields['feed_id'],
            item_no=event.fields['item_no'],
            comment_id=event.fields['comment_id'],
        )

    def action(self, event):
        comment_id = event.fields['comment_id']
        self.handle_common(event)
        if comment_id:
            self.handle_comment(event)
        else:
            self.handle_post(event)


names = list(locals().keys())
__all__ = [name for name, obj in locals().items() if isinstance(obj, type) and issubclass(obj,
                                                 (Event, Handler))]
