# -*- coding: utf-8 -*-

import logging

from at.common.utils import *
from at.common import utils
from at.aux_.Mailing import Mailing
from at.aux_ import Community
from at.aux_ import UserEvents
from at.aux_.RecentActions import RecentActionsLock, IncrementCounter, DecrementCounter
from at.pump.Query import Handler, Event
from at.pump import HandlerRegistry


_log = logging.getLogger ('pump')


class InsertRecentHandler(Handler):

    class Event(Event):
        field_names = ['person_id', 'feed_id', 'item_no', 'comment_id', 'store_time']

    def action(self, event):
        person_id, feed_id, item_no, comment_id, store_time = event
        with get_connection() as connection:
            store_time = utils.parse_isoformat(store_time)
            IncrementCounter(connection, person_id, 1)
            UserEvents.EventList(person_id).push(
                    UserEvents.CommentNotification\
                        (feed_id, item_no,  comment_id, store_time))
            return event.get_human_repr()


class CleanupCommentHistoryHandler(Handler):

    class Event(Event):
        field_names = ['uid', 'feed_id', 'item_no', 'comment_id']

    def action(self, event):
        uid, feed_id, item_no, comment_id = event
        with RecentActionsLock(uid):
            events = UserEvents.EventList(uid, ['CommentNotification']).filter(feed_id=feed_id, item_no=item_no, comment_id=comment_id)
            if not len(events):
                return True
            if len(events) > 1:
                _log.warn('Unexpectedly %d CommentNotifications for cleanup' % len(events))
            for e in events:
                e.state.add('seen')
                e.state.add('deleted')
            with get_connection() as connection:
                    DecrementCounter(connection, uid, len(events))
                    events.save()
        return event.get_human_repr()


class CleanupLikesHandler(Handler):
    class Event(Event):
        field_names = ['feed_id', 'item_no', 'comment_id']

    def action(self, event):
        feed_id, item_no, comment_id = event
        filter_args = {
            'feed_id': feed_id,
            'item_no': item_no
        }
        if comment_id != 0:
            # Для того, чтобы при удалении поста грохнуть информеры о комментах к посту
            filter_args['comment_id'] = comment_id
        with RecentActionsLock(feed_id):
            events = UserEvents.EventList(feed_id, ['Like']).filter(
                **filter_args
            )
            for e in events:
                e.state.add('seen')
                e.state.add('deleted')
            events.save()

        return event.get_human_repr()


class DeletedCommentHandler(Handler):
    """
    Хэндлер для обработки удалённых комментариев.
    Затирает все следы и прячет все хвосты, как будто никакого комментария и не было.
    Нужно вызывать для каждого поль
    """
    class Event(Event):
        field_names = ['person_id', 'feed_id', 'item_no', 'comment_id']

    def action(self, event):
        _, feed_id, item_no, comment_id = event
        for uid in Mailing.GetMailingTargets(
                feed_id, item_no, comment_id, unfiltered_uids=True):
            HandlerRegistry.put_event(
                'CleanupCommentHistoryHandler',
                uid=uid,
                feed_id=feed_id,
                item_no=item_no,
                comment_id=comment_id,
            )
        return event.get_human_repr()

                            
class RecentCommentsHandler(Handler):
    class Event(Event):
        field_names = ['person_id', 'feed_id', 'item_no', 'comment_id', 'store_time']

    def action(self, event):
        person_id, feed_id, item_no, comment_id, store_time = event
        try:
            for uid in Mailing.GetMailingTargets(feed_id, item_no, comment_id,
                    unfiltered_uids=True) - set([person_id]):
                HandlerRegistry.put_event(
                    'InsertRecentHandler',
                    person_id=uid,
                    feed_id=feed_id,
                    item_no=item_no,
                    comment_id=comment_id,
                    store_time=store_time,
                )
        except exceptions.NotFound:
            # ??? откуда вообще такое исключение?..
            pass
        return event.get_human_repr()


class ClubNotifyHandler(Handler):
    class Event(Event):
        field_names = ['feed_id', 'action_type', 'item_no',
                'squealer_id', 'moderator_id']

    def action(self, event):
        feed_id, action_type, item_no, sender_id, moder_id = event
        if moder_id == sender_id:
            return 'own'
        UserEvents.EventList(moder_id).push(UserEvents.ClubNotification(action_type, sender_id, feed_id, item_no))
        return event.get_human_repr()



class ClubCheckHandler(Handler):
    class Event(Event):
        field_names = [
            'feed_id', 'action_type', 'item_no',
            'comment_id', 'moderator_id'
        ]

    def action(self, event):
        _log.info('Checking a club note: %s' % (event,))
        feed_id, action_type, item_no, _, mod_id = event
        # XXX not good having cycle in a handler (not atomical); rewrite with separate events for each one
        for uid in ([mod_id] if mod_id else Community.get_alive_moderators(feed_id)):
            events = UserEvents.EventList(uid, ['ClubNotification'], ['seen']).\
                    filter(feed_id=feed_id, item_no=item_no, type=action_type)
            for e in events:
                e.state.update(set(['seen']))
            events.save()
        return event.get_human_repr()


class LikeNotificationHandler(Handler):
    """ Inform users about the new likes on their posts. """
    class Event(Event):
        field_names = ['author_id', 'feed_id', 'item_no', 'comment_id', 'uid']

    def action(self, event):
        author_id, feed_id, item_no, comment_id, uid = event
        _log.debug("New like event for user %s on post %s.%s by %s" % (author_id, feed_id, item_no, uid))

        # Creating a new notification for the user
        try:
            assert UserEvents.EventList(author_id).push(
                UserEvents.Like(uid,
                                feed_id,
                                item_no,
                                comment_id)), \
                "Can't push like event into EventList"
        except UserEvents.ItemHasGone:
            return True
        return event.get_human_repr()

