# -*- coding: utf-8 -*-



import logging
import types

from at.common import exceptions
from at.common.utils import get_connection, is_community_id
from at.common import groups
from at.common.groups import GroupType, AccessLevel
from at.aux_ import Relations
from at.aux_.models import Person
from at.aux_ import feeds
from at.pump.Query import Handler, Event


_log = logging.getLogger(__name__)


class CommunityStatHandler(Handler):
    accepts_new_style = True
    class Event(Event):
        field_names = ['feed_id']

    def action(self, event):
        feed_id = event.fields['feed_id']
        try:
            feed = Person.get_person(feed_id)
        except exceptions.NotFound:
            msg = 'Feed %s not found'
            _log.warning(msg, feed_id)
            return event.get_human_repr() + ' not found'

        if not is_community_id(feed_id) or feed.status != 'normal':
            msg = "ignore feed - %d"
            _log.warning(msg, feed_id)
            return event.get_human_repr() + ' ignore'

        with get_connection() as conn:
            # количество публичных (без премодерируемых) постов
            # и время последнего комментария
            get_posts_sql = 'SELECT COUNT(*), MAX(last_updated), MAX(store_time) FROM Posts '\
                    'WHERE person_id=%d AND deleted=0 AND access_group in (%s)'
            access_types = ",".join(
                [str(groups.get_int_access(at)) for at in [
                    AccessLevel.PUBLIC,
                    AccessLevel.MEMBERS,
                ]]
            )
            cur = conn.execute(get_posts_sql % (feed_id, access_types))
            posts_count, last_comment_time, last_post_time = cur.fetchone()

            members_count = Relations.get_total_relations_counters(feed_id, [GroupType.MEMBER])[GroupType.MEMBER]

        with get_connection() as conn:
            update_sql = """
                REPLACE INTO CommunitiesStat
                    (feed_id, members_count, posts_count, last_comment_time, last_post_time)
                VALUES (%d, %d, %d, '%s', '%s')"""
            args =  (feed_id, members_count, posts_count,
                    last_comment_time or '0000-00-00 00:00:00',
                    last_post_time or '0000-00-00 00:00:00')
            _log.debug(update_sql % args)
            conn.execute(update_sql % args)
        return True


class ScorePostHandler(Handler):

    class Event(Event):
        field_names = ['feed_id', 'item_no']

    def action(self, event):
        feed_id, post_no = event
        scorer = feeds.PostScorer(feed_id, post_no)
        with get_connection() as conn:
            try:
                scorer.score(conn)
            except exceptions.NotFound:
                _log.warning("ScorePostHandler: failed to load post %s.%s.", feed_id, post_no)


names = list(locals().keys())
__all__ = [name for name, obj in locals().items() if isinstance(obj, type) and issubclass(obj,
                                                 (Event, Handler))]
