# coding: utf-8

from collections import defaultdict
import logging

from django.conf import settings

from ids.registry import registry

from at.aux_.models import Person
from at.common import exceptions, oauth
from at.aux_.BlogsAuxiliary import BlogsAuxiliaryMix
from at.common import ServiceAccess
from at.common.ServiceAccess import AFFILIATIONS, FORCED_DEPARTMENTS, FORCED_GROUPS
from at.common import utils
from at.common.tvm2_client import get_service_ticket
from at.common.utils import get_connection
from at.aux_ import Profile
from at.aux_ import models

log = logging.getLogger(__name__)


class SyncError(AssertionError):
    pass


def dismiss_blogs(uids):
    log.info('Updating blog status for dismissed personnel.')

    aux = BlogsAuxiliaryMix()
    for uid in uids:
        log.debug('dismissing %s' % uid)
        try:
            aux.DismissBlog(uid)
        except Exception:
            log.exception('Exception while blog dismiss for UID {}'.format(uid))


def restore_blogs(uids):
    log.info('Restoring blogs for returnees.')

    aux = BlogsAuxiliaryMix()
    for uid in uids:
        log.debug('resurrecting %s' % uid)
        try:
            utils.getAuthInfo(uid)
        except exceptions.NotFound as ex:
            log.warning("User uid not found in staff %s\n%s", uid, ex)
            continue
        try:
            aux.ResurrectBlog(uid)
        except Exception:
            log.exception('Exception while blog resurrection for UID {}'.format(uid))


def create_blogs(uids, fetched_persons):
    log.info('Creating blogs for new personnel.')

    errors = []
    for uid in uids:
        try:
            Profile.maybe_create_blog(uid, fetched_persons[uid])
        except Exception as exc:
            errors.append((uid, str(exc)))

    if errors:
        log.critical("Can't create for %s users: %s", len(errors), errors)
        raise SyncError(errors)


def update_profile(ai, data):
    try:
        Profile.UpdateSource(ai, data)
    except exceptions.NotFound:
        log.info('user %s not found' % data['feed_id'])


def update_memorial(uids):
    for uid in uids:
        try:
            person = models.Person.get_by(person_id=uid)
            person.status = 'memorial'
            person.save()
        except Exception:
            log.exception("Error while updating memorials for UID {}".format(uid))


def update_gender(uids, gender):
    log.info('Synchronizing genders: %s' % (gender,))

    ai = utils.getGodAuthInfo()

    for uid in uids:
        data = {
            'feed_id': uid,
            'sex': gender
        }

        update_profile(ai, data)


def update_birthdays(birthdays):
    log.info('Synchronizing birthdays.')

    sql = 'update bday set bday = %s where person_id = %s'

    with get_connection() as conn:
        for person_id, bday in birthdays:
            log.debug('setting %d birthday: %s' % (person_id, bday))
            try:
                conn.execute(sql, (bday, person_id))
            except Exception:
                log.exception('Error while birthday sync for Person: {} with Birthday: {}'.format(person_id, bday))


def delete_birthdays(birthdays):
    log.info('Cleaning birthdays.')

    sql = 'DELETE FROM bday WHERE person_id IN %(person_ids)s'

    with get_connection() as conn:
        if birthdays:
            try:
                conn.execute(sql, person_ids=tuple(birthdays))
            except Exception:
                id_to_remove = ', '.join(str(person_id) for person_id in birthdays)
                log.exception('Error while removing birthdays for persons: {}'.format(id_to_remove))


def update_emails(emails):
    log.info('Synchronizing emails')
    sql = """
    UPDATE persons
    SET
        email = %s
    WHERE
            email != %s
        AND
            person_id = %s
    """
    with get_connection() as conn:
        for person_id, email in emails:
            try:
                if email and email.strip():
                    log.debug('setting %d email: %s' % (person_id, email))
                    conn.execute(sql, (email, email, person_id))
            except Exception:
                log.exception('Error while updating email for Person: {} with Email: {}.'.format(person_id, email))


def update_names(names, dismissed):
    log.info("Synchronizing names")
    ai = utils.getGodAuthInfo()
    for uid, names in names.items():
        if uid not in dismissed:
            try:
                title, title_eng = names
                data = {
                    'feed_id': uid,
                    'title': utils.force_str(title),
                    'title_eng': utils.force_str(title_eng)
                }
                update_profile(ai, data)
            except Exception:
                log.exception("update_names failed for uid %s" % uid)


def update_access_rights(person_access):
    log.info("Updating access rights")
    try:
        Person.objects.filter(person_id__in=person_access[True]).update(has_access=True)
        Person.objects.filter(person_id__in=person_access[False]).update(has_access=False)
    except Exception:
        log.exception('Error while access sync')


class Database(object):
    def __init__(self):
        self.all = set()
        self.by_status = defaultdict(set)
        self.by_gender = defaultdict(set)
        self.birthdays = set()
        self.load()

    def load(self):
        with get_connection() as conn:
            query = 'select person_id, status, sex, bday from persons inner join bday using(person_id) where person_id < %s'
            cursor = conn.execute(query, (settings.COMMUNITY_START_ID,))
            for uid, status, gender, bday in cursor:
                self.add(uid, status, gender, bday)

    def add(self, uid, status, gender, bday):
        self.all.add(uid)
        self.by_status[status].add(uid)
        self.by_gender[gender].add(uid)
        self.birthdays.add((uid, bday))

    @property
    def males(self):
        return self.by_gender['man']

    @property
    def females(self):
        return self.by_gender['woman']

    @property
    def normal(self):
        return self.by_status['normal']

    @property
    def dismissed(self):
        return self.by_status['dismissed']


SYNC_STAFF_FIELDS = (
    'uid',
    'official.is_dismissed',
    'personal.gender',
    'personal.birthday',
    'name',
    'work_email',
    'memorial',
)


class Staff(object):
    def __init__(self, load_limit):
        self.person_access = {True: [], False: []}
        self.by_status = defaultdict(set)
        self.by_gender = defaultdict(set)
        self.birthdays = set()
        self.birthdays_to_delete = set()
        self.names = {}
        self.emails = set()
        self.memorial = set()
        self.fetched_persons = {}
        self._persons = registry.get_repository(
            'staff', 'person', user_agent='at',
            service_ticket=get_service_ticket('staff'),
            retries=5, timeout=15,
        )
        self.load_limit = load_limit
        self.load()

    def load(self):
        fields_str = ','.join(
            set(SYNC_STAFF_FIELDS) |
            set(ServiceAccess.ACCESS_CHECK_STAFF_FIELDS)
        )
        # Тяжелый запрос, выбирает всех-всех
        users = self._persons.getiter(lookup={'_fields': fields_str, '_limit': self.load_limit})

        for person in users:
            if not person['uid']:
                continue

            uid = int(person['uid'])
            is_dismissed = person['official']['is_dismissed']
            self.by_status[is_dismissed].add(uid)
            if person.get('memorial') is not None:
                self.memorial.add(uid)

            self.person_access[self.calculate_person_accessibility(person)].append(uid)

            if is_dismissed:
                continue

            self.by_gender[person['personal']['gender']].add(uid)
            self.names[uid] = (
                '%s %s' % (
                    person['name']['first']['ru'],
                    person['name']['last']['ru']
                ),
                '%s %s' % (
                    person['name']['first']['en'],
                    person['name']['last']['en']
                )
            )
            self.emails.add((uid, person["work_email"]))
            bday = person['personal']['birthday']
            parsed_bday = None
            if bday is not None:
                try:
                    parsed_bday = self.parse_birthday(bday)
                except ValueError:
                    msg = "Failed to parse birthday for %d: %s"
                    log.info(msg % (uid, bday))

            if parsed_bday is not None:
                self.birthdays.add((uid, parsed_bday))
            else:
                self.birthdays_to_delete.add(uid)

            self.fetched_persons[uid] = person

    def parse_birthday(self, bday):
        year, month, day = list(map(int, bday.split('-')))
        return '%02d-%02d' % (month, day)

    @staticmethod
    def calculate_person_accessibility(person):
        if person is None:
            return False

        if person['official'].get('is_dismissed', False):
            return False

        affiliation = person['official'].get('affiliation')

        if affiliation is None:
            return False

        if affiliation in AFFILIATIONS:
            return True

        department_group = person.get('department_group')

        if department_group is None:
            return False

        ancestors = department_group.get('ancestors')
        department_group_url = department_group.get('url')

        if ancestors is None or department_group_url is None:
            return False

        departments = set(
            [anc['url'] for anc in ancestors] + [department_group_url]
        )

        if departments.intersection(FORCED_DEPARTMENTS):
            return True

        groups = set([group['group']['url'] for group in person.get('groups', [])])

        if groups.intersection(FORCED_GROUPS):
            return True

        return False

    @property
    def current(self):
        return self.by_status[False]

    @property
    def former(self):
        return self.by_status[True]

    @property
    def males(self):
        return self.by_gender['male']

    @property
    def females(self):
        return self.by_gender['female']

    @property
    def dismissed(self):
        return self.by_status[True]


def sync(load_limit=settings.STAFF_LOAD_LIMIT):
    db = Database()
    staff = Staff(load_limit)

    update_emails(staff.emails)
    dismiss_blogs(db.normal & staff.former)
    restore_blogs(db.dismissed & staff.current)
    create_blogs(staff.current - db.all, staff.fetched_persons)
    update_gender(staff.males - db.males - staff.dismissed, 'man')
    update_gender(staff.females - db.females - staff.dismissed, 'woman')
    update_access_rights(staff.person_access)
    update_birthdays(staff.birthdays - db.birthdays)
    delete_birthdays(staff.birthdays_to_delete & db.birthdays)
    update_names(staff.names, staff.dismissed)
    update_memorial(staff.memorial)
