# coding: utf-8

import os
import sys
import logging

from lxml import etree

import yenv

from at.common import utils
from at.aux_ import entries
from at.common import exceptions
from at.common import Types
from at.common import dbswitch


FORMAT = '%(asctime)-15s %(levelname)s %(message)s'
formatter = logging.Formatter(FORMAT)

log = logging.getLogger()
log.handlers = []
stream_handler = logging.StreamHandler()
file_handler = logging.FileHandler('/tmp/%s.log' % os.path.basename(__file__))
stream_handler.setFormatter(formatter)
file_handler.setFormatter(formatter)

log.addHandler(stream_handler)
log.addHandler(file_handler)
log.setLevel(logging.DEBUG)


AT_HOST = yenv.choose_key_by_type({
    'production': 'clubs.at.yandex-team.ru',
    'testing': 'clubs.at.test.yandex-team.ru',
    'development': 'clubs.sibirev.at.dev.yandex-team.ru',
})


def find_entries_by_type_name(post_type):
    type_id = Types.ItemTypes().by_name(post_type)['id']
    sql = '''
    SELECT person_id, post_no, comment_id
        FROM Comments
        WHERE comment_type = %s
        AND deleted = 0
    '''
    with utils.get_connection() as conn:
        row_set = conn.execute(sql, (type_id,))
        return list(map(tuple, row_set))


def collect_entries(post_type):
    entries_ids = find_entries_by_type_name(post_type)
    data = {}
    for entry_id in entries_ids:
        data[entry_id] = {
            'xml_data': get_xml_data(entry_id),
            'entry': get_entry(entry_id)
        }
    return data


def convert_entries(type, mode='convert'):
    entries_list = find_entries_by_type_name(type)
    data, errors = [], []

    if mode == 'print':
        for entry_id in entries_list:
            print_entry_url(entry_id)
        return

    for entry_id in entries_list:
        person_id, item_no, comment_id = entry_id
        print_entry_url(entry_id)
        try:
            converted = convert_entry(entry_id)
        except Exception:
            feed_is_deleted = is_feed_deleted(person_id)
            post_is_deleted = is_post_deleted(person_id, item_no)
            if feed_is_deleted:
                msg = error(entry_id, 'Club is deleted')
                log.error(msg)
            elif post_is_deleted:
                msg = error(entry_id, 'Post is deleted')
                log.error(msg)
            else:
                msg = error(entry_id)
                log.exception(msg)
            errors.append((entry_id, msg))
        else:
            data.append(converted)

    covert_deleted_entries(source_type=type)
    delete_entry_type(type=type)
    return data, errors


def get_entry(id):
    god_ai = utils.getGodAuthInfo()
    try:
        entry = entries.load_entry(*id)
    except exceptions.NotFound:
        return
    entry.content_type = 'text/wiki'
    return entry


def convert_entry(id, target_type='text', with_debug=True):
    entry = get_entry(id)
    if entry.entry_type == 'wishlist':
        convert_wishlist(id, entry)
    entry.entry_type = target_type
    entry.save()

    if with_debug:
        print_entry_url(id)
        return get_entry(id)


def get_entry_url(id):
    return 'https://%s/%s/%s/%s' % tuple([AT_HOST] + list(id))


def error(id, extra=''):
    return 'Error with %s (%s). %s' % (id, get_entry_url(id), extra)


def print_entry_url(id):
    print(get_entry_url(id))


def is_feed_deleted(feed_id):
    sql = """
SELECT status
FROM persons
WHERE person_id = %s
    """
    with utils.get_connection() as conn:
        feeds = list(conn.execute(sql, (feed_id,)))
        if feeds:
            return feeds[0][0] == 'deleted'
    return False


def is_post_deleted(feed_id, item_no):
    sql = """
SELECT deleted
FROM Posts
WHERE person_id = %s AND post_no = %s
    """
    with utils.get_connection() as conn:
        feeds = list(conn.execute(sql, (feed_id, item_no)))
        if feeds:
            return bool(feeds[0][0])
    return False


def get_xml(id):
    feed_id, item_no, comment_id = id
    sql = '''
    SELECT xml FROM EntryXmlContent
    WHERE
      feed_id = :feed_id AND
      post_no = :item_no AND
      comment_id = :comment_id
    '''
    with dbswitch.root_rw_session() as conn:
        row = conn.execute(sql, {
            'feed_id': feed_id,
            'item_no': item_no,
            'comment_id': comment_id,
        }).fetchone()
        return row[0]


def get_xml_data(id):
    xml = get_xml(id)
    node = etree.fromstring(xml)

    return {
        'raw': xml,
        'title': _fetch(node, 'content/title'),
        'title_original': _fetch(node, 'content/title-original'),
        'body': _fetch(node, 'content/body'),
        'url': _fetch(node, 'meta/url'),
        'image': _fetch(node, 'meta/image'),
    }


def _fetch(node, xpath, attr='text'):
    elem = node.find(xpath)
    return elem is not None and getattr(elem, attr)


def convert_wishlist(id, entry):
    xml_data = get_xml_data(id)
    url, img = None, None
    if xml_data['title_original']:
        url = xml_data['title_original']
    elif xml_data['url']:
        url = xml_data['url']

    if xml_data['image']:
        img = xml_data['image']

    if url:
        entry.body = entry.body_original + '\n' + url

    if img:
        entry.body = entry.body_original + '\n' + img


def covert_deleted_entries(source_type, target_type='text'):
    source_type_id = Types.ItemTypes().by_name(source_type)['id']
    target_type_id = Types.ItemTypes().by_name(target_type)['id']

    sql_comments = '''
        UPDATE Comments
        SET comment_type = :target_type
        WHERE comment_type = :source_type
    '''
    sql_posts = '''
        UPDATE Posts
        SET post_type = :target_type
        WHERE post_type = :source_type
    '''
    with dbswitch.root_rw_session() as conn:
        for sql in (sql_comments, sql_posts):
            conn.execute(sql, {
                'source_type': source_type_id,
                'target_type': target_type_id,
            })


def delete_entry_type(type):
    sql = '''
        DELETE FROM ItemTypes
        WHERE name = :type
    '''
    with dbswitch.root_rw_session() as conn:
        conn.execute(sql, {'type': type})


if __name__ == '__main__':
    if len(sys.argv) == 1:
        print('entry type required')
    else:
        convert_entries(*sys.argv[1:])
