# coding: utf-8



import os
import sys
import logging

from at.common import utils
from at.aux_ import entries
from at.common import exceptions
from at.common import Types
from at.common import dbswitch


FORMAT = '%(asctime)-15s %(levelname)s %(message)s'
formatter = logging.Formatter(FORMAT)

log = logging.getLogger()
log.handlers = []
stream_handler = logging.StreamHandler()
file_handler = logging.FileHandler('/tmp/%s.log' % os.path.basename(__file__))
stream_handler.setFormatter(formatter)
file_handler.setFormatter(formatter)

log.addHandler(stream_handler)
log.addHandler(file_handler)
log.setLevel(logging.DEBUG)


def ensure_entry(func):
    def wrapper(entry_or_id, *args, **kwargs):
        if isinstance(entry_or_id, tuple):
            entry = get_entry(id=entry_or_id)
        else:
            entry = entry_or_id
        return func(entry, *args, **kwargs)
    return wrapper


def ensure_id(func):
    def wrapper(entry_or_id, *args, **kwargs):
        if not isinstance(entry_or_id, tuple):
            id = get_id(entry=entry_or_id)
        else:
            id = entry_or_id
        return func(id, *args, **kwargs)
    return wrapper


@ensure_id
def get_entry_url(id):
    import yenv
    AT_URL = yenv.choose_key_by_type({
        'development': 'https://clubs.sibirev.at.dev.yandex-team.ru/%s/%s/%s',
        'testing': 'https://clubs.at.test.yandex-team.ru/%s/%s/%s',
        'production': 'https://clubs.at.yandex-team.ru/%s/%s/%s',
    })
    return AT_URL % id


@ensure_id
def print_entry_url(id):
    print(get_entry_url(id))


def get_id(entry):
    return entry.feed_id, entry.item_no, entry.comment_id


def find_entries_by_type_name(post_type):
    type_id = Types.ItemTypes().by_name(post_type)['id']
    sql = '''
    SELECT person_id, post_no, 0
        FROM Posts
        WHERE post_type = %s
        AND deleted = 0
    ORDER BY person_id, store_time
    '''
    with utils.get_connection() as conn:
        row_set = conn.execute(sql, (type_id,))
        return list(map(tuple, row_set))


def collect_entries():
    rules_ids = find_entries_by_type_name('rules')
    descriptions_ids = find_entries_by_type_name('description')

    feeds = {}
    for rule_id in rules_ids:
        feed_id, item_no, comment_id = rule_id
        feed_data = feeds.setdefault(feed_id, {})
        feed_data['rules'] = get_entry(rule_id)
    for descr_id in descriptions_ids:
        feed_id, item_no, comment_id = descr_id
        feed_data = feeds.setdefault(feed_id, {})
        feed_data['description'] = get_entry(descr_id)

    return feeds


def convert_entries(dry=True):
    feeds_ids = collect_entries()
    for feed_id, data in list(feeds_ids.items()):
        convert_feed_data(feed_data=data, dry=dry, pin=True)

    # добиваем те, что без
    feeds_ids = collect_entries()
    for feed_id, data in list(feeds_ids.items()):
        convert_feed_data(feed_data=data, dry=dry, pin=False)


def drop_types():
    convert_deleted_entries(source_type='rules')
    convert_deleted_entries(source_type='description')
    delete_entry_type(type='rules')
    delete_entry_type(type='description')


def convert_feed_data(feed_data, dry=True, pin=True):
    rules = feed_data.get('rules')
    description = feed_data.get('description')

    if description:
        normalize_post(description)
        if pin:
            description.pinned_after = 0
        if not dry:
            description.save()

    if rules:
        normalize_post(rules)
        if pin:
            if description:
                rules.pinned_after = description.item_no
            else:
                rules.pinned_after = 0
        if not dry:
            rules.save()


def print_feed_data(feed_data):
    description = feed_data.get('description')
    rules = feed_data.get('rules')

    if description:
        print('[description]', get_entry_url(description))
        print('content_type', description.content_type)
        print('body', description._body)
        print('body-original', description.body_original)
        print('pinned_after', description.pinned_after)
    if rules:
        print('[rules]', get_entry_url(rules))
        print('content_type', rules.content_type)
        print('body', rules._body)
        print('body-original', rules.body_original)
        print('pinned_after', rules.pinned_after)


def normalize_post(post):
    body, is_html = get_editable_piece(post)

    if is_html:
        body = '<#' + body + '#>'

    post.entry_type = 'text'
    post.type = 'text'
    post.content_type = 'text/wiki'
    post.body = body


def get_editable_piece(entry):
    if not entry:
        return '', False

    if entry.content_type != 'text/plain':
        piece = entry.body_original or entry._body
        is_html = is_html_body(piece)
        return piece, is_html
    else:
        if entry.body_original:
            is_html = is_html_body(entry.body_original)
            if is_html:
                piece = entry.body_original.replace('\n', '<br />')
            else:
                piece = entry.body_original
            return piece, is_html
        else:
            return entry._body, is_html_body(entry._body)


def is_html_body(body):
    if body.startswith('<#'):
        return False
    if '<' in body:
        return True
    return False


@ensure_id
def get_entry(id):
    god_ai = utils.getGodAuthInfo()
    try:
        entry = entries.load_entry(*id)
    except exceptions.NotFound:
        return
    except Exception:
        log.exception('load entry %s (%s) failed', id, get_entry_url(id))
        return
    return entry


def is_feed_deleted(feed_id):
    sql = """
SELECT status
FROM persons
WHERE person_id = %s
    """
    with utils.get_connection() as conn:
        feeds = list(conn.execute(sql, (feed_id,)))
        if feeds:
            return feeds[0][0] == 'deleted'
    return False


def is_post_deleted(feed_id, item_no):
    sql = """
SELECT deleted
FROM Posts
WHERE person_id = %s AND post_no = %s
    """
    with utils.get_connection() as conn:
        feeds = list(conn.execute(sql, (feed_id, item_no)))
        if feeds:
            return bool(feeds[0][0])
    return False


@ensure_id
def get_xml(id):
    feed_id, item_no, comment_id = id
    sql = '''
    SELECT xml FROM EntryXmlContent
    WHERE
      feed_id = :feed_id AND
      post_no = :item_no AND
      comment_id = :comment_id
    '''
    with dbswitch.root_rw_session() as conn:
        row = conn.execute(sql, {
            'feed_id': feed_id,
            'item_no': item_no,
            'comment_id': comment_id,
        }).fetchone()
        return row[0]


def convert_deleted_entries(source_type, target_type='text'):
    source_type_id = Types.ItemTypes().by_name(source_type)['id']
    target_type_id = Types.ItemTypes().by_name(target_type)['id']

    sql_comments = '''
        UPDATE Comments
        SET comment_type = :target_type
        WHERE comment_type = :source_type
    '''
    sql_posts = '''
        UPDATE Posts
        SET post_type = :target_type
        WHERE post_type = :source_type
    '''
    with dbswitch.root_rw_session() as conn:
        for sql in (sql_comments, sql_posts):
            conn.execute(sql, {
                'source_type': source_type_id,
                'target_type': target_type_id,
            })


def delete_entry_type(type):
    sql = '''
        DELETE FROM ItemTypes
        WHERE name = :type
    '''
    with dbswitch.root_rw_session() as conn:
        conn.execute(sql, {'type': type})


if __name__ == '__main__':
    args = sys.argv[1:]
    if not args:
        convert_entries(dry=True)
    elif args == ['convert']:
        convert_entries(dry=False)
    elif args == ['drop']:
        drop_types()
