import logging

from django.conf import settings
from django.utils.translation import ugettext_lazy as _
from django.utils.functional import cached_property

from rest_framework.exceptions import APIException
from rest_framework import status


from .logic.errors import pop_non_field_rf_errors
from intranet.audit.src.core.utils.strings import strings_to_list

logger = logging.getLogger(__name__)

REST_API_ERROR_MAP = {
    'ValidationError': _('Ошибка валидации'),
}


def get_error_message(error):
    class_name = error.__class__.__name__
    return REST_API_ERROR_MAP.get(class_name, class_name)


def rest_api_error_to_dict(error):
    """
    @type error: errors.RestApiError
    @rtype: dict
    """
    data = {
        'debug_message': error.debug_message,
        'error_code': error.error_code,
        'level': 'ERROR',
    }
    if error.errors is not None:
        data['errors'] = error.errors
    if error.non_field_messages:
        data['message'] = error.non_field_messages
    return data


class RestApiError(APIException):
    """
    Базовая ошибка.
    У наследников должен быть определен уникальный error_code.
    Не плодите уникальные error_code, нужно стараться использовать существующие.
    Потому что использовать бекэнд удобнее, когда error_code мало.
    Часто хватает ошибки InvalidDataSentError.
    """
    STATUSES = status

    status_code = STATUSES.HTTP_409_CONFLICT

    # Сообщение об ошибке для разработчика, не для пользователя.
    # Указывать обязательно.
    # Показывать клиенту не нужно.
    debug_message = ''

    # Словарь вида "имя поля из запроса" - "список локализованных строк о том, как исправить ошибку"
    _field_errors = None

    # Список локализованных сообщений об ошибке, которые не относятся ни к одному полю из запроса.
    # Пример: сервис находится в рид-онли.
    _non_field_messages = None

    def __init__(self, errors_or_message=None, non_field_messages=None, debug_message=None):
        """
        @param debug_message: строка на английском языке, сообщение об ошибке, которое нужно для отладки.
        @param errors_or_message: словарь с информацией о деталях ошибки: ключ – строка, индентификатор детали,
                       значение - строка, пояснение, в чём проблема с этой деталью. Или строка,
                       тогда она считается non_field_messages, и non_field_messages нужно не указывать.
        @type errors_or_message: dict|str
        @type non_field_messages: str|list
        @param non_field_messages: Сообщение в виде строки или список сообщений об ошибках.
        Следующие вызовы одинаковы:
        raise RestApiError('error')
        raise RestApiError(non_field_messages='error')
        Т.е. если у вас нет errors, но есть non_field_messages, то достаточно указать non_field_messages
        первым аргументом.
        Про rest_framework:
        Если в rest_framework бросить ValidationError в методе validate сериализатора, то она
        не будет относиться ни к одному из полей. Все ошибки, которые возникли в сериализаторе
        и не относятся к конкретным полям, попадут из errors_or_message в non_field_messages.
        """
        extra_non_field_messages = []
        if errors_or_message is not None:
            if isinstance(errors_or_message, dict):
                # для удобства написания кода, при написании легко забыть
                # что вместо InvalidDataSentError({'field': 'error'})
                # надо писать InvalidDataSentError({'field': ['error']})
                # они даже при чтении кода слабо различимы.
                for key, value in errors_or_message.items():
                    if hasattr(value, 'capitalize'):
                        errors_or_message[key] = strings_to_list(value)
                extra_non_field_messages.extend(pop_non_field_rf_errors(
                    errors_or_message
                ))
                self._field_errors = errors_or_message
            elif hasattr(errors_or_message, 'capitalize'):  # what possibly it is if not a string?
                if non_field_messages:
                    # message для пользователя можно передавать
                    # либо в errors_or_message, либо в non_field_messages,
                    # но не сразу в обоих.
                    raise ValueError(
                        'You are trying to instantiate {0} with two conflicting messages "{1}" and "{2}"'.format(
                            self.__class__.__name__,
                            errors_or_message,
                            non_field_messages
                        ))
                non_field_messages = errors_or_message
            else:
                raise ValueError(
                    'errors should be a dict or a string, you gave me {0}'.format(type(errors_or_message)))

        if non_field_messages is not None:
            extra_non_field_messages.extend(strings_to_list(
                non_field_messages))

        if extra_non_field_messages:
            self._non_field_messages = extra_non_field_messages

        self.debug_message = debug_message or self.debug_message
        if not self.debug_message:
            logger.warning('Debug message is empty, please specify. Class "%s"',
                           self.__class__.__name__)

        assert hasattr(self, 'error_code'), 'Class %s must have "error_code" attribute' % type(self)

    def __repr__(self):
        field_errors = ''
        if self.errors:
            field_errors = '; '.join(
                field + ': ' + ', '.join(errors)
                for field, errors in self.errors.items()
            )
        non_field_errors = ''
        if self.non_field_messages:
            non_field_errors = ','.join(self.non_field_messages)
        return 'field errors: "{field}", non-field_errors: "{nonfield}", debug message: "{debug}"'.format(
            field=field_errors,
            nonfield=non_field_errors,
            debug=self.debug_message,
        )

    def lazy_strings_to_unicode(self, item):
        """
        Привести все ленивые объекты к unicode
        Ленивые джанго-строки из ugettext на печати выглядят как
        django.utils.functional.__proxy__, это неудобно для repr.
        """
        if item is None:
            return item
        if isinstance(item, dict):
            for key in item:
                item[key] = self.lazy_strings_to_unicode(item[key])
            return item
        elif hasattr(item, '__iter__'):
            return [value.replace('\n', '\\n') for value in map(str, item)]
        else:
            return str(item).replace('\n', '\\n')

    @cached_property
    def errors(self):
        return self.lazy_strings_to_unicode(self._field_errors)

    @cached_property
    def non_field_messages(self):
        return self.lazy_strings_to_unicode(self._non_field_messages)

    def __str__(self):
        return self.__repr__()


class UnhandledException(RestApiError):
    status_code = RestApiError.STATUSES.HTTP_500_INTERNAL_SERVER_ERROR
    error_code = 'UNKNOWN_ERROR'
    debug_message = 'Unhandled exception'

    @property
    def _non_field_messages(self):
        return [
            # Translators:
            #  ru: Неизвестная ошибка, обратитесь за помощью к %(email_contact)s, пожалуйста
            #  en: Unknown error, contact %(email_contact)s for help, please
            _('Unknown error, contact %(email_contact)s for help, please') % {'email_contact': settings.SUPPORT_CONTACT}]


class YauthError(RestApiError):
    status_code = RestApiError.STATUSES.HTTP_401_UNAUTHORIZED
    error_code = 'AUTHENTICATION_ERROR'
    debug_message = 'Invalid token provided'


class BadRequestError(RestApiError):
    status_code = RestApiError.STATUSES.HTTP_409_CONFLICT
    error_code = 'BAD_REQUEST'
    debug_message = 'Request with incorrect parameters was made'


class NotFoundError(RestApiError):
    status_code = RestApiError.STATUSES.HTTP_404_NOT_FOUND
    error_code = 'NOT FOUND'
    debug_message = 'Request with invalid pk was made'


class BackendError(RestApiError):
    status_code = RestApiError.STATUSES.HTTP_500_INTERNAL_SERVER_ERROR
    error_code = 'BACKEND_ERROR'
    debug_message = 'Error acquired on backend'


class AccessError(RestApiError):
    status_code = RestApiError.STATUSES.HTTP_403_FORBIDDEN
    error_code = 'ACCESS_ERROR'
    debug_message = 'You don\'t have rights for this action'
