import collections

from rest_framework import permissions

from intranet.audit.src.api_v1.views.helpers import can_change


METHOD_ACTIONS = {
    'PATCH': 'change',
    'GET': 'view',
    'OPTIONS': 'view',
    'HEAD': 'view',
    'POST': 'add',
    'PUT': 'change',
    'DELETE': 'delete',
}

REVERSE_ACTIONS = collections.defaultdict(set)
for method, perm in METHOD_ACTIONS.items():
    REVERSE_ACTIONS[perm].add(method)


class AuditBasePermission(permissions.BasePermission):

    def has_permission(self, request, view):
        if request.user.is_superuser:
            return True
        model = view.serializer_class.Meta.model
        model_name = model.__name__
        model_app = model._meta.app_label
        method = request.method
        action = METHOD_ACTIONS[method]
        perm = '{}.{}_{}'.format(model_app, action, model_name.lower())
        return request.user.has_perm(perm)


class AuditViewDependentPermission(permissions.BasePermission):

    def has_permission(self, request, view):
        if request.user.is_superuser:
            return True
        needed_perm = view.get_perm_to_check(request)
        return request.user.has_perm(needed_perm)


class AuditImmutablePermission(permissions.BasePermission):

    def has_object_permission(self, request, view, obj):
        return (
            request.method in permissions.SAFE_METHODS
            or can_change(request.user, obj)
        )
