from rest_framework.pagination import CursorPagination, _reverse_ordering
from django.db.models import Model
from django.db.models import Min

BOOL_FIELDS = {
    'key_control',
    'antifraud',
    'mrc',
    'roll_forward',
    'need_to_be_aggregated',
}


class AuditPagination(CursorPagination):
    ordering = 'id'
    cursor_query_param = 'page'
    page_size_query_param = 'per_page'
    max_page_size = 200

    def _get_position_from_instance(self, instance, ordering):
        """
        Доработка оригинального метода
        https://github.com/encode/django-rest-framework/blob/master/rest_framework/pagination.py#L772
        для поддержки пагинации при сортировке по связанным
        полям вида order_by=risk__name

        TODO Добавить возможность корректного перехода на предыдущую страницу
        некоторые наброски - https://paste.yandex-team.ru/376482
        """
        field_data = ordering[0].lstrip('-').split('__')
        attr = instance

        for field_name in field_data:
            if isinstance(attr, dict):
                attr = attr[field_name]

            elif isinstance(attr, Model):
                attr = getattr(attr, field_name)

            elif attr.__class__.__name__ == 'ManyRelatedManager':
                return

        if isinstance(attr, Model):
                attr = getattr(attr, 'id')

        return str(attr)

    def paginate_queryset(self, queryset, request, view=None):
        self.page_size = self.get_page_size(request)
        if not self.page_size:
            return None

        self.base_url = request.build_absolute_uri()
        self.ordering = self.get_ordering(request, queryset, view)

        self.cursor = self.decode_cursor(request)
        if self.cursor is None:
            (offset, reverse, current_position) = (0, False, None)
        else:
            (offset, reverse, current_position) = self.cursor
        order_query = dict()
        order_by = list()
        for counter, order_value in enumerate(self.ordering):
            key = '{}_{}'.format('orderfunc', counter)

            if any(field in order_value.strip('-') for field in BOOL_FIELDS):
                order_by.append(order_value)
                continue

            if order_value.startswith('-'):
                    order_value = order_value[1:]
                    param = '-{}'.format(key)
            else:
                    param = key

            order_query[key] = Min(order_value)
            order_by.append(param)

        queryset = queryset.annotate(**order_query).order_by(*order_by, '-id')

        if current_position is not None:
            order = self.ordering[0]
            is_reversed = order.startswith('-')
            order_attr = order.lstrip('-')

            if self.cursor.reverse != is_reversed:
                kwargs = {order_attr + '__lt': current_position}
            else:
                kwargs = {order_attr + '__gt': current_position}

            queryset = queryset.filter(**kwargs)
        results = list(queryset[offset:offset + self.page_size + 1])
        self.page = list(results[:self.page_size])

        if len(results) > len(self.page):
            has_following_position = True
            following_position = self._get_position_from_instance(results[-1], self.ordering)
        else:
            has_following_position = False
            following_position = None

        if reverse:
            self.page = list(reversed(self.page))

        if reverse:
            self.has_next = (current_position is not None) or (offset > 0)
            self.has_previous = has_following_position
            if self.has_next:
                self.next_position = current_position
            if self.has_previous:
                self.previous_position = following_position
        else:
            self.has_next = has_following_position
            self.has_previous = (current_position is not None) or (offset > 0)
            if self.has_next:
                self.next_position = following_position
            if self.has_previous:
                self.previous_position = current_position

        if (self.has_previous or self.has_next) and self.template is not None:
            self.display_page_controls = True

        return self.page
