from datetime import date

from django.apps import apps
from django.utils.translation import ugettext
from rest_framework import status
from rest_framework.renderers import JSONRenderer
from rest_framework.response import Response

from intranet.audit.src.api_v1.errors import (
    AccessError,
    BackendError,
    BadRequestError,
    NotFoundError,
)
from intranet.audit.src.api_v1.serializers import ControlPlanSerializer, ControlTestDetailSerializer
from intranet.audit.src.api_v1.views.helpers import (
    get_actions,
    get_controlplan_available_statuses,
    get_controltest_available_statuses,
    make_actions_copy,
)
from intranet.audit.src.api_v1.views.view import APIView
from intranet.audit.src.core.models import ControlPlan, ControlTest
from intranet.audit.src.users.dao.user import get_stated_persons_by_uids_from_staff


SERIALIZERS = {
    'controlplan': ControlPlanSerializer,
    'controltest': ControlTestDetailSerializer,
}


class ActionsView(APIView):
    renderer_classes = (JSONRenderer,)

    def init_data(self, obj_class, pk):
        try:
            self.model_class = apps.get_model('core', obj_class)
        except LookupError:
            raise BadRequestError(ugettext(
                message='Request with wrong obj_class was made: %s' % obj_class)
            )
        try:
            self.object = self.model_class.objects.get(pk=pk)
        except self.model_class.DoesNotExist:
            raise NotFoundError()
        self.statuses = getattr(self.model_class, 'STATUSES', None)

    def get(self, request, obj_class, pk):
        self.init_data(obj_class, pk)
        response = get_actions(self.request.user, self.object)
        return Response(response, status=status.HTTP_200_OK)

    def post(self, request, obj_class, pk):
        self.init_data(obj_class, pk)
        if not self.request.user.has_perm('core.change_%s' % obj_class):
            raise AccessError('You dont have permissions to perform this action')

        action = self.get_data_from_request('action')
        if action.startswith('to_'):
            result = self.change_status(action, obj_class)
            obj_actions = get_actions(self.request.user, self.object)
            response = {
                'actions': dict(make_actions_copy(self.object), **obj_actions),
            }
            if result:
                response.update(result)
        elif action == 'add_version':
            statuses = self.object._meta.model.STATUSES
            if not self.object.status == statuses.active:
                raise BadRequestError('You can not add new version for current status')
            self.object.status = statuses.archived
            self.object.test_period_finished = (
                self.get_data_from_request('data', 'test_period_finished', required=False)
                or date.today()
            )
            self.object.save()

            response = {
                'status': 'ok',
            }
        else:
            raise BadRequestError('Request with unknown action was made')

        return Response(response, status=status.HTTP_200_OK)

    def change_status(self, needed_status, obj_class):
        available_statuses = None
        if isinstance(self.object, ControlPlan):
            available_statuses = get_controlplan_available_statuses(
                user=self.request.user,
                obj=self.object,
                obj_actions=dict(),
            )
        if isinstance(self.object, ControlTest):
            available_statuses = get_controltest_available_statuses(
                user=self.request.user,
                obj=self.object,
                obj_actions=dict(),
            )
        has_available_statuses = (
            available_statuses
            and needed_status in available_statuses
            and available_statuses[needed_status]
        )
        if not has_available_statuses:
            raise BadRequestError('You can not set this status for current object')

        needed_status = needed_status.split('_', 1)[1]

        if needed_status == self.statuses.review:
            reviewers_uids = self.get_data_from_request('data', 'reviewer_id').split(',')
            reviewers_objects = get_stated_persons_by_uids_from_staff(
                *reviewers_uids,
                request=self.request,
            ).values()
            if len(reviewers_uids) != len(reviewers_objects):
                raise BackendError('Not succeed in creating reviewers, please, try again')

            self.object.reviewer.clear()
            self.object.reviewer.add(*reviewers_objects)

        if isinstance(self.object, ControlTest) and needed_status == self.statuses.active:
            self.object.review_date = (
                self.get_data_from_request('data', 'review_date', required=False)
                or date.today()
            )
        if isinstance(self.object, ControlPlan) and needed_status == self.statuses.archived:
            self.object.test_period_finished = (
                self.get_data_from_request('data', 'test_period_finished', required=False)
                or date.today()
            )

        self.object.status = needed_status
        self.object.save()

        return {obj_class: SERIALIZERS[obj_class](self.object).data}
