from django.db import transaction
from django.utils.translation import ugettext
from django.core import urlresolvers

from rest_framework.response import Response
from rest_framework.renderers import JSONRenderer

from django.shortcuts import redirect

from intranet.audit.src.core import models
from intranet.audit.src.core.dao.controltest import create_controltests_by_plans
from intranet.audit.src.users.dao.user import get_stated_persons_by_uids_from_staff
from .view import APIView
from ..errors import BadRequestError, BackendError, AccessError
from .mixins import ViewDependentPermissionsMixin


class RunControlPlanView(ViewDependentPermissionsMixin, APIView):
    """
    Создает новые control_test по переданному
    control_plan, поддерживает создание сразу
    нескольких control_test, если передано
    несколько control_plan
    """
    renderer_classes = (JSONRenderer,)

    @transaction.atomic
    def post(self, request):

        if not request.user.has_perm('core.add_controltest'):
            # Translators: Используется в ответе апи в случае если пользователь
            # не имеет прав на создание controltest
            raise AccessError(ugettext('You don\'t have rights to create ControlTest'))

        obj_pks = self.get_param_from_request(request, 'obj_pks')
        testers = None
        testers_ids = self.get_param_from_request(request, 'assign_to')
        if testers_ids:
            testers_ids = testers_ids.split(',')
            testers = get_stated_persons_by_uids_from_staff(*testers_ids,
                                                            request=request,
                                                            )

            if not testers or not all(tester_ids in testers for tester_ids in testers_ids):
                # Translators: Используется в ответе апи в случае если произошла
                # ошибка при создании пользователя
                raise BackendError(ugettext('Could not create persons: "{}"'.format(testers_ids)))
            testers = testers.values()

        if not obj_pks:
            # Translators: Используется в ответе апи в случае если в запросе не
            # были переданы obj_pks
            raise BadRequestError(ugettext('You should pass obj_pks in request'))

        if isinstance(obj_pks, str):
            obj_pks = obj_pks.split(',')

        created_tests = self.create_tests_by_plans(*obj_pks,
                                                   author=request.user,
                                                   testers=testers,
                                                   )

        if len(obj_pks) == 1:
            link_to_object = self.obj_admin_link(created_tests[int(obj_pks[0])])
            return redirect(link_to_object)

        data = {'control_test_created': len(obj_pks)}
        return Response(data)

    def create_tests_by_plans(self, *plan_ids, author, testers):
        control_plans = models.ControlPlan.objects.filter(pk__in=plan_ids)
        if not control_plans or len(control_plans) != len(plan_ids):
            # Translators: Используется в ответе апи в случае если в запросе были
            # переданы некорректные obj_pks
            raise BadRequestError(ugettext('Check id\'s correction - ControlPlans does not exists'))

        control_tests = create_controltests_by_plans(
            *control_plans,
            author=author,
            testers=testers,
        )

        return control_tests

    def obj_admin_link(self, obj):
        return urlresolvers.reverse("admin:core_{}_change".format(obj.__class__.__name__.lower()),
                                    args=[obj.id],
                                    )

    def get_perm_to_check(self, request):
        return 'core.add_controltest'
