from django.utils.translation import ugettext
from django.db import transaction

from rest_framework.response import Response

from ..errors import BadRequestError
from .view import APIView
from .mixins import ViewDependentPermissionsMixin
from intranet.audit.src.core import models


class ControlStepMoveView(ViewDependentPermissionsMixin, APIView):
    model = models.ControlStep
    available_actions = {'move-after', 'move-before', }

    def get_perm_to_check(self, request):
        return 'core.change_controlstep'

    def post(self, request, pk, action, related_pk):
        pk, related_pk = int(pk), int(related_pk)
        if pk == related_pk:
            # Translators: Используется в ответе апи в случае
            # если в запросе был переданы одинаковые id
            raise BadRequestError(ugettext('Objects pk\'s should be different'))

        if action not in self.available_actions:
            # Translators: Используется в ответе апи в случае
            # если в запросе был передан неверный action
            raise BadRequestError(ugettext('Bad action was passed'))

        with transaction.atomic():
            try:
                obj = self.model.objects.select_related('control_test').get(pk=pk)
            except self.model.DoesNotExist:
                # Translators: Используется в ответе апи в случае
                # если в запросе был передан неверный id
                raise BadRequestError(ugettext('Request with invalid pk was made'))

            if not obj.control_test:
                # Translators: Используется в ответе апи в случае
                # если в запросе был передан id без привязанного ControlTest
                raise BadRequestError(ugettext('There is no related ControlTest'))

            controlstep_order = list(obj.control_test.get_controlstep_order())

            if not all(obj_id in controlstep_order for obj_id in [pk, related_pk]):
                # Translators: Используется в ответе апи в случае
                # если в запросе был переданы id относящиеся к разным ControlTest
                # или если переданного related_pk не существует
                raise BadRequestError(ugettext(('Request with pk\'s related '
                                                'to different ControlTest was made or '
                                                'passed related_pk does not exists')))
            controlstep_order.remove(pk)
            related_place = controlstep_order.index(related_pk)
            if action == 'move-after':
                controlstep_order.insert(related_place + 1, pk)
            elif action == 'move-before':
                controlstep_order.insert(related_place, pk)

            obj.control_test.set_controlstep_order(controlstep_order)
        return Response(data={'status': 'ok'})
