from intranet.audit.src.api_v1 import serializers
from intranet.audit.src.api_v1.errors import BadRequestError
from intranet.audit.src.api_v1.logic.permissions import AuditImmutablePermission
from intranet.audit.src.api_v1.views.metadata import ControlPlanMetadata
from intranet.audit.src.api_v1.views.mixins import RelatedMixin, SendToDraftMixin
from intranet.audit.src.api_v1.views.view import RetrieveUpdateAPIView
from intranet.audit.src.core import models
from intranet.audit.src.files.models import File


class RetrieveUpdateSafeDeleteAPIView(RetrieveUpdateAPIView):
    """
    Базовая вьюха для отображения, редактирования
    и безопасного удаления объектов.
    Под безопасным удалением подразумевается,
    что используемые объекты удалять нельзя.
    """
    def _is_related_object_exists(self, instance):
        for rel in instance._meta.related_objects:
            related_manager = getattr(instance, rel.get_accessor_name())
            if related_manager.exists():
                return True
        return False

    def perform_destroy(self, instance):
        if self._is_related_object_exists(instance):
            raise BadRequestError()
        return super(RetrieveUpdateSafeDeleteAPIView, self).perform_destroy(instance)


class ControlPlanDetailView(SendToDraftMixin, RelatedMixin, RetrieveUpdateAPIView):

    STATUSES = models.ControlPlan.STATUSES

    metadata_class = ControlPlanMetadata
    queryset = models.ControlPlan.objects.all()
    permission_classes = (AuditImmutablePermission,)
    serializer_class = serializers.ControlPlanSerializer
    select_related = ('control',)
    prefetch_related = (
        'process',
        'system',
        'service',
        'risk',
        'account',
        'legal',
        'business_unit',
        'assertion',
        'owner',
        'process__parent',
        'account__parent',
        'reviewer',
    )


class ControlTestDetailView(SendToDraftMixin, RelatedMixin, RetrieveUpdateAPIView):

    STATUSES = models.ControlTest.STATUSES

    queryset = models.ControlTest.objects.all()
    permission_classes = (AuditImmutablePermission,)
    serializer_class = serializers.ControlTestDetailSerializer
    select_related = ('control_plan', 'control_plan__control')
    prefetch_related = (
        'tester',
        'evidence',
        'reviewer',
        'deficiency',
        'control_plan__process',
        'deficiency__author',
        'control_plan__system',
        'control_plan__service',
        'control_plan__risk',
        'control_plan__account',
        'control_plan__legal',
        'control_plan__business_unit',
        'control_plan__assertion',
        'control_plan__owner',
        'control_plan__process__parent',
        'control_plan__account__parent',
        'control_plan__reviewer',
        'controlstep_set',
        'controlstep_set__file',
        'control_test_ipe',
        'control_test_ipe__attach',
        'control_test_ipe__ipe',
        'control_test_ipe__ipe__evidence',
        'control_test_ipe__ipe__system',
        'control_test_ipe__ipe__author',
        'control_test_ipe__ipe__reviewer',
    )


class AccountDetailView(RelatedMixin, RetrieveUpdateSafeDeleteAPIView):
    queryset = models.Account.objects.all()
    serializer_class = serializers.AccountSerializer
    select_related = ('parent',)


class AssertionDetailView(RetrieveUpdateSafeDeleteAPIView):
    queryset = models.Assertion.objects.all()
    serializer_class = serializers.AssertionSerializer


class BusinessUnitDetailView(RetrieveUpdateSafeDeleteAPIView):
    queryset = models.BusinessUnit.objects.all()
    serializer_class = serializers.BusinessUnitSerializer


class ControlDetailView(RetrieveUpdateSafeDeleteAPIView):
    queryset = models.Control.objects.all()
    serializer_class = serializers.ControlSerializer


class ControlStepDetailView(RelatedMixin, RetrieveUpdateAPIView):
    queryset = models.ControlStep.objects.all()
    serializer_class = serializers.ControlStepDetailSerializer
    select_related = ('control_test',)
    prefetch_related = ('file',)


class DeficiencyDetailView(RelatedMixin, RetrieveUpdateAPIView):
    queryset = models.Deficiency.objects.all()
    serializer_class = serializers.DeficiencySerializer
    select_related = ('author',)

    prefetch_related = (
        'control_test__tester',
        'control_test__evidence',
        'control_test__deficiency',
        'control_test__control_plan__process',
        'control_test__control_plan__system',
        'control_test__control_plan__service',
        'control_test__control_plan__risk',
        'control_test__control_plan__reviewer',
        'control_test__control_plan__account',
        'control_test__control_plan__legal',
        'control_test__control_plan__business_unit',
        'control_test__control_plan__assertion',
        'control_test__control_plan__owner',
        'control_test__control_plan__process__parent',
        'control_test__control_plan__account__parent',
        'control_test__controlstep_set',
        'control_test__control_plan',
        'control_test__reviewer',
        'control_test',
    )


class DeficiencyGroupDetailView(RelatedMixin, RetrieveUpdateAPIView):

    queryset = models.DeficiencyGroup.objects.all()
    serializer_class = serializers.DeficiencyGroupSerializer
    select_related = (
        'author',
    )


class IPEDetailView(RelatedMixin, RetrieveUpdateAPIView):
    queryset = models.IPE.objects.all()
    serializer_class = serializers.IPEDetailSerializer

    select_related = ('author', 'system')
    prefetch_related = (
        'control_test_ipe__control_test__tester',
        'control_test_ipe__control_test__evidence',
        'control_test_ipe__control_test__deficiency',
        'control_test_ipe__control_test__control_plan__process',
        'control_test_ipe__control_test__control_plan__system',
        'control_test_ipe__control_test__control_plan__reviewer',
        'control_test_ipe__control_test__control_plan__service',
        'control_test_ipe__control_test__control_plan__risk',
        'control_test_ipe__control_test__control_plan__account',
        'control_test_ipe__control_test__control_plan__legal',
        'control_test_ipe__control_test__control_plan__business_unit',
        'control_test_ipe__control_test__control_plan__assertion',
        'control_test_ipe__control_test__control_plan__owner',
        'control_test_ipe__control_test__control_plan__process__parent',
        'control_test_ipe__control_test__control_plan__account__parent',
        'control_test_ipe__control_test__controlstep_set',
        'control_test_ipe__control_test__control_plan',
        'control_test_ipe__control_test__reviewer',
        'evidence',
        'control_test_ipe',
        'reviewer',
    )


class LegalDetailView(RetrieveUpdateSafeDeleteAPIView):
    queryset = models.Legal.objects.all()
    serializer_class = serializers.LegalSerializer


class RiskDetailView(RetrieveUpdateSafeDeleteAPIView):
    queryset = models.Risk.objects.all()
    serializer_class = serializers.RiskSerializer


class ServiceDetailView(RelatedMixin, RetrieveUpdateSafeDeleteAPIView):
    queryset = models.Service.objects.all()
    serializer_class = serializers.ServiceSerializer


class SystemDetailView(RelatedMixin, RetrieveUpdateSafeDeleteAPIView):
    queryset = models.System.objects.all()
    serializer_class = serializers.SystemSerializer


class ProcessDetailView(RelatedMixin, RetrieveUpdateSafeDeleteAPIView):
    queryset = models.Process.objects.all()
    serializer_class = serializers.ProcessSerializer
    select_related = ('parent',)


class FileDetailView(RetrieveUpdateAPIView):
    queryset = File.objects.all()
    serializer_class = serializers.FileDetailSerializer


class ControlTestIPEDetailView(RelatedMixin, RetrieveUpdateAPIView):
    queryset = models.ControlTestIPE.objects.all()
    serializer_class = serializers.ControlTestIPEDetailSerializer

    select_related = (
        'ipe',
        'control_test',
        'ipe__system',
        'ipe__author',
    )
    prefetch_related = (
        'attach',
        'ipe__reviewer',
        'control_test__reviewer',
    )
