import aiohttp
import asyncio
import io
import logging
import openpyxl

from django.core.urlresolvers import reverse
from django.http import HttpResponse
from django.utils.functional import cached_property
from django.utils.translation import ugettext
from openpyxl.drawing.image import Image
from openpyxl.styles import Alignment, Font
from openpyxl.writer.excel import save_virtual_workbook

from intranet.audit.src.api_v1.errors import BadRequestError
from intranet.audit.src.api_v1.views import (
    ControlPlanDetailView,
    ControlTestDetailView,
    IPEDetailView,
)
from intranet.audit.src.api_v1.views.mixins import (
    FilterMixin,
    RelatedMixin,
    ViewDependentPermissionsMixin,
)
from intranet.audit.src.api_v1.views.view import APIView
from intranet.audit.src.core.models import (
    ControlPlan,
    ControlTest,
    Deficiency,
    DeficiencyGroup,
    IPE,
)
from intranet.audit.src.core.resources import (
    ControlPlanResource,
    ControlTestResource,
    DeficiencyGroupResource,
    DeficiencyResource,
    IPEResource,
    get_field_help_text,
)
from intranet.audit.src.files.models import ALLOW_EXCEL


logger = logging.getLogger(__name__)


class ExportView(RelatedMixin, ViewDependentPermissionsMixin, FilterMixin, APIView):
    """
    View выгрузки объектов в xlsx

    Для добавления новых моделей в выгрузку следует добавить запись вида
    '<model_name>': {'model': <класс_модели>,
                    'related': <класс_вьюхи_с_нужными_select/prefetch_related>,
                    'files': <название_поля_с_файлами>,
                    'resource': <класс_ресурса_для_осуществления_экспорта>,
                    'wide_columns': <iterable_с_названиями_широких_колонок>
                    }
    где model,resource - обязательны
    """
    MODELS_MAP = {
        'controlplan': {
            'model': ControlPlan,
            'related': ControlPlanDetailView,
            'resource': ControlPlanResource,
            'wide_columns': {
                'контроль',
                'риск',
                'процесс',
                'description',
                'evidence',
                'regulation',
                'comment',
            },
            'short_columns': {
                'ключевой контроль',
                'antifraud',
                'status',
            },
        },

        'controltest': {
            'model': ControlTest,
            'related': ControlTestDetailView,
            'files': 'evidence',
            'resource': ControlTestResource,
            'wide_columns': {
                'контроль',
                'риск',
                'процесс',
                'sampling',
                'evidence',
                'comment',
                'steps',
                'steps comments',
                'threshold for investigation',
                get_field_help_text(ControlTest, 'how_precision_is_affected'),
                get_field_help_text(ControlTest, 'how_management_identifies'),
            },
            'short_columns': {
                'steps statuses',
                'design efficiency',
                'operational efficiency',
                'ключевой контроль',
                'status',
                'mrc',
                'roll-forward',
            },
        },

        'ipe': {
            'model': IPE,
            'related': IPEDetailView,
            'files': 'evidence',
            'resource': IPEResource,
            'wide_columns': {
                'ipe',
                'source data',
                'comment',
                'report logic',
            },
            'short_columns': {
                'type',
                'appliance',
                'created',
                'status',
            },
        },
        'deficiency': {
            'model': Deficiency,
            'resource': DeficiencyResource,
            'wide_columns': {
                'контроль',
                'риск',
                'процесс',
                'short description',
                'full description',
                'ticket',
                'mitigating factors',
                'comment',
                'root_cause_of_the_control_deficiency',
                'indication_of_other_deficiencies',
                'factors_in_evaluating_severity',
                'application_controls',
                'impact_of_gitc_deficiency',
            },
            'short_columns': {
                'ключевой контроль',
                'design efficiency',
                'operational efficiency',
                'state',
                'created',
                'potential impact',
                'need_to_be_aggregated',
            },
        },
        'deficiencygroup': {
            'model': DeficiencyGroup,
            'resource': DeficiencyGroupResource,
        },
    }

    @cached_property
    def prepared_select_related(self):
        return self.get_related_from_view('select_related')

    @cached_property
    def prepared_prefetch_related(self):
        return self.get_related_from_view('prefetch_related')

    def get_related_from_view(self, related_type):
        view_with_related = self.model_data.get('related')
        if view_with_related is not None:
            return getattr(view_with_related, related_type)
        return tuple()

    def get_queryset(self):
        queryset = super().get_queryset()

        obj_pks = self.request.query_params.get('obj_pks')
        if obj_pks:
            obj_pks = obj_pks.split(',')
            queryset = queryset.filter(pk__in=obj_pks)
        return queryset

    def get_perm_to_check(self, request):
        return 'users.export_excel'

    def get(self, request, obj_class):
        model_data = self.MODELS_MAP.get(obj_class)
        if not model_data:
            raise BadRequestError(ugettext('Bad obj_class was passed'))

        self.model_data = model_data
        self.model = model_data['model']

        queryset = self.remove_duplicate(self.get_queryset())
        file_set = None

        base_data = model_data['resource']().export(queryset=queryset).xlsx

        base_book = io.BytesIO(base_data)
        result_book = openpyxl.load_workbook(base_book)
        self._add_format(result_book)
        if file_set:
            loop = asyncio.get_event_loop()
            task = asyncio.ensure_future(self._add_files(result_book, file_set))
            loop.run_until_complete(task)

        response = HttpResponse(
            content=save_virtual_workbook(result_book),
            content_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        )
        response['Content-Disposition'] = 'attachment; filename={}.xlsx'.format(obj_class)

        return response

    def _get_url_for_file(self, file):
        link_to_file = reverse(
            viewname='files:file',
            kwargs={
                'pk': file.id,
            },
        )
        full_url = '{protocol}://{host}{url}'.format(
            protocol='https' if self.request.is_secure() else 'http',
            host=self.request.META['HTTP_HOST'],
            url=link_to_file,
        )
        return full_url

    async def _add_files(self, book, file_set):
        async with aiohttp.ClientSession(raise_for_status=True) as session:
            tasks = [
                asyncio.ensure_future(self._add_file(file, book, session))
                for file in file_set
            ]
            await asyncio.wait(tasks)

    async def _add_file(self, file, book, session):
        file_sheet = book.create_sheet(title=file.name)
        file_sheet['A1'].font = Font(bold=True)
        file_sheet['A1'] = file.name

        if file.content_type not in ALLOW_EXCEL:
            message = (
                'File in format ({}) that cannot be attached to XLS.'
                'Click on the link below to see it.'
            )
            file_sheet['A2'] = message.format(file.content_type)

            file_sheet['A3'].hyperlink = self._get_url_for_file(file)
            file_sheet['A3'].value = file.name
            return

        async with session.get(file.file.url, timeout=5) as response:
            try:
                file_content = await response.content.read()
            except (aiohttp.ClientError, asyncio.TimeoutError):
                logger.exception('Got error from storage, file: "%s"', file.name)
                message = 'Cannot get file from storage'
                file_sheet['A2'] = message
            else:

                file_obj = io.BytesIO(file_content)

                img = Image(file_obj)
                img.anchor(file_sheet.cell('A2'))
                file_sheet.add_image(img)

    def _add_format(self, book):
        work_sheet = book.worksheets[0]
        wide_columns = self.model_data.get('wide_columns', set())
        short_columns = self.model_data.get('short_columns', set())
        for col in work_sheet.iter_cols(min_row=1, max_row=1):
            header = col[0]
            header_value = header.value.lower()
            if header_value in wide_columns:
                width = 56
            elif header_value in short_columns:
                width = 14
            else:
                width = 28
            work_sheet.column_dimensions[header.column_letter].width = width

        for row in work_sheet.iter_rows(min_row=1):
            for cell in row:
                cell.alignment = Alignment(
                    wrap_text=True,
                    horizontal='left',
                    vertical='top',
                )
