import io
import logging
import asyncio
import aiohttp
import itertools
import zipfile

from django.http import HttpResponse
from django.utils.translation import ugettext
from rest_framework.response import Response

from intranet.audit.src.core.models import ControlTest
from intranet.audit.src.files.models import File
from .view import APIView
from .mixins import (
    FilterMixin,
    ViewDependentPermissionsMixin,
)

from ..errors import BadRequestError, NotFoundError

logger = logging.getLogger(__name__)

MODELS_MAP = {
    'controltest': {
        'model': ControlTest,
        'files_to_export': [
            'evidence',
            'controlstep_set__file',
            'control_test_ipe__attach',
        ]
    }
}


class ExportFilesView(ViewDependentPermissionsMixin, FilterMixin, APIView):

    def get_perm_to_check(self, request):
        obj_class = request.parser_context['kwargs'].get('obj_class')
        return 'core.view_{}'.format(obj_class)

    def get(self, request, obj_class, pk):
        model_data = MODELS_MAP.get(obj_class)
        if not model_data:
            # Translators: Используется в ответе апи в случае
            # если в запросе был передан неверный obj_class
            raise BadRequestError(ugettext('Bad obj_class was passed'))

        file_objects_set = self.get_files_objects_for_model(model_data, pk)

        if file_objects_set:
            loop = asyncio.get_event_loop()
            response_content = loop.run_until_complete(
                self.get_response_content(file_objects_set)
            )
            response = HttpResponse(response_content, content_type="application/x-zip-compressed")
            response['Content-Disposition'] = 'attachment; filename={}_{}.zip'.format(obj_class, pk)
        else:
            response = Response({'detail': 'No attached files found'})

        return response

    async def get_response_content(self, file_objects_set):
        in_memory_zip = io.BytesIO()
        zip_file = zipfile.ZipFile(in_memory_zip, "w")
        async with aiohttp.ClientSession(raise_for_status=True) as session:
            tasks = [asyncio.ensure_future(self._get_file(file,
                                                          session,
                                                          ))
                     for file in file_objects_set]
            for future in asyncio.as_completed(tasks):
                file_data = await future
                if file_data:
                    file_obj, file_name = file_data
                    zip_file.writestr(file_name, file_obj.read())

        zip_file.close()
        return in_memory_zip.getvalue()

    def get_files_objects_from_path(self, file_model, path, i=0):
        while not hasattr(file_model, 'model'):
            file_model = getattr(file_model, path[i])
            i += 1

        if file_model.model == File:
            return file_model.all()
        else:
            return itertools.chain.from_iterable(
                self.get_files_objects_from_path(obj, path, i)
                for obj in file_model.all()
            )

    def get_files_objects_for_model(self, model_data, pk):
        files_objects = []
        try:
            obj = model_data['model'].objects.get(pk=pk)
        except model_data['model'].DoesNotExist:
            raise NotFoundError()

        for file_path in model_data['files_to_export']:
            file_path = file_path.split('__')
            files_objects.extend(self.get_files_objects_from_path(obj, file_path))

        return set(files_objects)

    async def _get_file(self, file, session):

        async with session.get(file.file.url, timeout=5) as response:
            try:
                file_content = await response.content.read()
            except (aiohttp.ClientError, asyncio.TimeoutError):
                logger.exception('Got error from storage, file: "%s"', file.name)
            else:
                file_obj = io.BytesIO(file_content)
                return file_obj, file.name
