import logging
import collections

from drf_multiple_model.views import ObjectMultipleModelAPIView

from .view import APIView
from ..serializers import (
    ControlForFilterSerializer,
    ProcessForFilterSerializer,
    ServiceForFilterSerializer,
    SystemForFilterSerializer,
    RiskForFilterSerializer,
)

from intranet.audit.src.core import models

logger = logging.getLogger(__name__)

ADDITIONAL_DATA = {
    'controlplan': {
        'status': [
            {'id': value, 'name': verbose_value}
            for value, verbose_value in models.ControlPlan.STATUSES
        ]
    },
    'deficiency': {
        'significance_evaluation': [
            {'id': value, 'name': verbose_value}
            for value, verbose_value in models.Deficiency.PROBABILITIES
        ]
    },
    'ipe': {
        'status': [
            {'id': value, 'name': verbose_value}
            for value, verbose_value in models.IPE.STATUSES
        ],
        'ipe_type': [
            {'id': value, 'name': verbose_value}
            for value, verbose_value in models.IPE.TYPES
        ],
        'appliance': [
            {'id': value, 'name': verbose_value}
            for value, verbose_value in models.IPE.APPLIANCES
        ],

    },
    'controltest': {
        'status': [
            {'id': value, 'name': verbose_value}
            for value, verbose_value in models.ControlTest.STATUSES
        ],
    },
}

FILTER_MODELS = {
    'ipe': {
        'system',
    },
}

FILTER_EXISTING = {
    'controlplan': {
        'system': 'controlplan',
        'control': 'controlplan',
        'service': 'controlplan',
        'process': 'parent_process__controlplan',
        'risk': 'controlplan',
    },
    'deficiency': {
        'system': 'controlplan__controltest__deficiency',
        'control': 'controlplan__controltest__deficiency',
        'service': 'controlplan__controltest__deficiency',
        'process': 'parent_process__controlplan__controltest__deficiency',
        'risk': 'controlplan__controltest__deficiency',
    },
    'controltest': {
        'system': 'controlplan__controltest',
        'control': 'controlplan__controltest',
        'service': 'controlplan__controltest',
        'process': 'parent_process__controlplan__controltest',
        'risk': 'controlplan__controltest',
    },
}


class FilterDataView(APIView, ObjectMultipleModelAPIView):
    pagination_class = None
    result_type = collections.OrderedDict

    querylist_base = [
        {'queryset': (models.Process.objects
                      .filter(process_type=models.Process.TYPES.root)
                      .prefetch_related('parent_process')
                      .order_by('name')
                      ),
         'serializer_class': ProcessForFilterSerializer,
         'model': 'process',
         },
        {'queryset': models.Control.objects.all().order_by('number'),
         'serializer_class': ControlForFilterSerializer,
         'model': 'control',
         },
        {'queryset': models.Risk.objects.all().order_by('number'),
         'serializer_class': RiskForFilterSerializer,
         'model': 'risk',
         },
        {'queryset': models.System.objects.all().order_by('name'),
         'serializer_class': SystemForFilterSerializer,
         'model': 'system',
         },
        {'queryset': models.Service.objects.all().order_by('name'),
         'serializer_class': ServiceForFilterSerializer,
         'model': 'service',
         },
    ]

    def get_querylist(self):
        querylist = self.querylist_base
        obj_class = self.request.parser_context['kwargs'].get('obj_class')
        if obj_class:
            filter_models = FILTER_MODELS.get(obj_class)
            if filter_models:
                querylist = [
                    model_data for model_data in querylist
                    if model_data['model'] in filter_models
                ]
        return querylist

    def format_results(self, results, request):
        obj_class = request.parser_context['kwargs'].get('obj_class')
        if obj_class:
            additional_data = ADDITIONAL_DATA.get(obj_class)
            if additional_data:
                results.update(additional_data)
            else:
                logger.warning('Request with not known obj_class was made "%s"', obj_class)
        return results

    def get_label(self, queryset, query_data):
        label = super().get_label(queryset, query_data)
        return label.lower()

    def load_queryset(self, query_data, request, *args, **kwargs):
        """
        Выводим только те сущности у которых есть связь с
        объектами класса из запроса, чтобы в фильтре не отображались
        сущности фильтрации по которым даст пустой результат
        """
        queryset = super().load_queryset(query_data, request, *args, **kwargs)
        obj_class = request.parser_context['kwargs'].get('obj_class')
        query_class = query_data['model']
        if obj_class:
            filter_query = FILTER_EXISTING.get(obj_class, {}).get(query_class)
            if filter_query:
                query = {'{}__isnull'.format(filter_query): False,
                         }
                queryset = queryset.filter(**query).distinct()
        return queryset
