from copy import copy

from intranet.audit.src.core.models import ControlPlan, ControlTest


BASE_ACTIONS = {
    "edit": False,
    "make_report": False,
}

CONTROLPLAN_ACTIONS = {
    "add_version": False,
    "add_test": False,
    "clone": False,

    "to_draft": False,
    "to_deleted": False,
    "to_review": False,
    "to_active": False,
    "to_archived": False,
}

CONTROLTEST_ACTIONS = {
    "download_attachments": False,
    "create_deficiency": False,
    "create_controltestipe": False,
    "change_controltestipe": False,
    "delete_controltestipe": False,
    "reorder_steps": False,
    "create_step": False,
    "delete_step": False,
    "change_step": False,

    "to_in_progress": False,
    "to_deleted": False,
    "to_review": False,
    "to_archived": False,
}


def set_true_for_action(*actions, action_map):
    for action in actions:
        action_map[action] = True
    return action_map


def make_actions_copy(obj):
    if isinstance(obj, ControlPlan):
        return copy(CONTROLPLAN_ACTIONS)
    if isinstance(obj, ControlTest):
        return copy(CONTROLTEST_ACTIONS)
    return {}


def can_change(user, obj):
    if isinstance(obj, ControlPlan):
        return can_change_controlplan(user, obj)
    if isinstance(obj, ControlTest):
        return can_change_controltest(obj)
    return True


def can_change_controlplan(user, obj):
    statuses = obj._meta.model.STATUSES
    if obj.status in (statuses.draft, statuses.review):
        return True
    elif obj.status == statuses.active:
        if not obj.has_related_tests:
            return True
        is_permitted = (
            obj.reviewer.filter(uid=user.uid).exists()
            and not obj.controltest_set.filter(status=statuses.archived).exists()
        )
        if is_permitted:
            return True
    return False


def can_change_controltest(obj):
    statuses = obj._meta.model.STATUSES
    return obj.status in (statuses.draft, statuses.review, statuses.active)


def get_base_actions(user, obj):
    obj_class = obj._meta.model_name
    obj_actions = copy(BASE_ACTIONS)

    if user.has_perm('core.view_%s' % obj_class):
        obj_actions = set_true_for_action('make_report', action_map=obj_actions)

    is_editable = (
        user.has_perm('core.change_%s' % obj_class)
        and can_change(user, obj)
    )
    if is_editable:
        obj_actions = set_true_for_action('edit', action_map=obj_actions)

    return obj_actions


def get_actions(user, obj):
    obj_actions = get_base_actions(user, obj)
    statuses = obj._meta.model.STATUSES

    if isinstance(obj, ControlPlan):
        if user.has_perm('core.add_controlplan'):
            obj_actions = set_true_for_action('clone', action_map=obj_actions)
        if obj.status == statuses.active:
            obj_actions = set_true_for_action('add_version', 'add_test', action_map=obj_actions)
        obj_actions = get_controlplan_available_statuses(user, obj, obj_actions)

    if isinstance(obj, ControlTest):
        if user.has_perm('core.view_controltest'):
            obj_actions = set_true_for_action('download_attachments', action_map=obj_actions)
        if obj.status == statuses.draft:
            obj_actions = set_true_for_action('create_deficiency', action_map=obj_actions)
        if obj.status in (statuses.draft, statuses.review):
            obj_actions = set_true_for_action(
                'create_controltestipe',
                'change_controltestipe',
                'delete_controltestipe',
                'reorder_steps',
                'create_step',
                'delete_step',
                'change_step',
                action_map=obj_actions,
            )
        obj_actions = get_controltest_available_statuses(user, obj, obj_actions)

    return obj_actions


def get_controlplan_available_statuses(user, obj, obj_actions):
    current_status = obj.status
    statuses = obj._meta.model.STATUSES
    available_statuses = []

    if current_status == statuses.review:
        available_statuses += (statuses.draft, statuses.deleted)
        if user.is_superuser or obj.reviewer.filter(uid=user.uid).exists():
            available_statuses += (statuses.active,)
    elif current_status == statuses.draft:
        available_statuses += (statuses.review, statuses.deleted)
    elif current_status == statuses.active:
        available_statuses.append(statuses.archived)
        if not obj.has_related_tests:
            available_statuses += (statuses.deleted, statuses.draft)

    available_statuses = ('to_%s' % status for status in available_statuses)
    return set_true_for_action(*available_statuses, action_map=obj_actions)


def get_controltest_available_statuses(user, obj, obj_actions):
    current_status = obj.status
    statuses = obj._meta.model.STATUSES
    available_statuses = []

    if current_status == statuses.review:
        available_statuses += (statuses.deleted,)
        if user.is_superuser or obj.reviewer.filter(uid=user.uid).exists():
            available_statuses += (statuses.active, statuses.draft)
    elif current_status == statuses.draft:
        available_statuses += (statuses.review, statuses.deleted)
    elif current_status == statuses.active:
        available_statuses += (statuses.archived, statuses.deleted, statuses.draft)

    available_statuses = ('to_%s' % status for status in available_statuses)
    return set_true_for_action(*available_statuses, action_map=obj_actions)
