from rest_framework import status
from rest_framework.response import Response

from django.core.exceptions import ValidationError
from django.conf import settings
from django.utils.translation import ugettext_lazy as _

from intranet.audit.src.api_v1 import serializers
from intranet.audit.src.core import models
from intranet.audit.src.core.dao.controltest import copy_controltest
from intranet.audit.src.files.models import File

from ..logic.files import get_first_file_from_request
from .view import ListCreateAPIView
from .metadata import ControlPlanMetadata
from .mixins import RelatedMixin


class ControlPlanListView(RelatedMixin, ListCreateAPIView):
    model = models.ControlPlan
    metadata_class = ControlPlanMetadata
    serializer_class = serializers.ControlPlanSerializer
    search_fields = (
        'control__number',
        'control__name',
        'risk__name',
        'risk__number',
        'account__name',
        'process__name',
        'business_unit__name',
        'legal__name',
        'system__name',
        'service__name',
        'owner__last_name',
        'owner__login',
        'owner__first_name',
        'reviewer__last_name',
        'reviewer__login',
        'reviewer__first_name',

        'id',
        'description',
        'evidence',
        'regulation',
        'comment',
    )

    select_related = ('control',)
    prefetch_related = (
        'process',
        'system',
        'service',
        'risk',
        'account',
        'legal',
        'business_unit',
        'assertion',
        'owner',
        'process__parent',
        'account__parent',
        'reviewer',
    )


class ControlTestListView(RelatedMixin, ListCreateAPIView):
    model = models.ControlTest
    serializer_class = serializers.ControlTestSerializer
    search_fields = (
        'tester__login',
        'tester__first_name',
        'tester__last_name',
        'deficiency__short_description',
        'deficiency__full_description',
        'deficiency__ticket_key',
        'deficiency__mitigating_factors',
        'control_plan__process__name',
        'control_plan__service__name',
        'control_plan__system__name',
        'control_plan__risk__name',
        'control_plan__control__name',
        'control_plan__account__name',
        'control_plan__legal__name',
        'control_plan__business_unit__name',
        'control_plan__assertion__name',
        'control_plan__control__number',
        'control_plan__risk__number',
        'control_plan__owner__last_name',
        'control_plan__owner__first_name',
        'control_plan__owner__login',
        'controlstep__step',

        'sampling',
        'evidence_comments',
        'status',
    )
    select_related = (
        'control_plan',
        'control_plan__control',
    )
    prefetch_related = (
        'tester',
        'evidence',
        'deficiency',
        'control_plan__process',
        'control_plan__process__parent',
        'control_plan__owner',
        'control_plan__system',
        'control_plan__service',
        'control_plan__risk',
        'control_plan__account',
        'control_plan__legal',
        'control_plan__business_unit',
        'control_plan__assertion',
        'control_plan__account__parent',
        'control_plan__reviewer',
        'controlstep_set',
        'deficiency__author',
        'reviewer',
    )

    def perform_create(self, serializer):
        control_plan = serializer.validated_data['control_plan']
        previous_test = (
            models.ControlTest.objects
            .filter(
                control_plan=control_plan,
                status__in=(
                    models.ControlTest.STATUSES.active,
                    models.ControlTest.STATUSES.archived,
                ),
            )
            .order_by('-test_period_finished')
            .first()
        )

        if previous_test:
            return copy_controltest(
                previous_test, serializer.validated_data,
                self.request.user.pk,
            )
        super().perform_create(serializer)

    def create(self, request, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        copy_existed = self.perform_create(serializer)
        if copy_existed:
            serializer = self.get_serializer(copy_existed)
        headers = self.get_success_headers(serializer.data)

        return Response(serializer.data, status=status.HTTP_201_CREATED, headers=headers)


class AccountListView(RelatedMixin, ListCreateAPIView):
    model = models.Account
    serializer_class = serializers.AccountSerializer
    search_fields = ('name',)
    select_related = ('parent',)


class AssertionListView(ListCreateAPIView):
    model = models.Assertion
    serializer_class = serializers.AssertionSerializer
    search_fields = ('name',)


class BusinessUnitListView(ListCreateAPIView):
    model = models.BusinessUnit
    serializer_class = serializers.BusinessUnitSerializer
    search_fields = ('name',)


class ControlListView(ListCreateAPIView):
    model = models.Control
    serializer_class = serializers.ControlSerializer
    search_fields = ('name', 'number')


class ControlStepListView(RelatedMixin, ListCreateAPIView):
    model = models.ControlStep
    serializer_class = serializers.ControlStepDetailSerializer
    search_fields = ('step', 'comment')
    select_related = ('control_test',)
    prefetch_related = ('file',)


class DeficiencyListView(RelatedMixin, ListCreateAPIView):
    model = models.Deficiency
    serializer_class = serializers.DeficiencySerializer
    search_fields = (
        'short_description',
        'full_description',
        'ticket_key',
        'mitigating_factors',
        'comment',

        'control_test__control_plan__process__name',
        'control_test__control_plan__service__name',
        'control_test__control_plan__system__name',
        'control_test__control_plan__account__name',
        'control_test__control_plan__legal__name',
        'control_test__control_plan__business_unit__name',
        'control_test__control_plan__assertion__name',
        'control_test__control_plan__risk__name',
        'control_test__control_plan__risk__number',
        'control_test__control_plan__control__name',
        'control_test__control_plan__control__number',
        'control_test__control_plan__owner__last_name',
        'control_test__control_plan__owner__first_name',
        'control_test__control_plan__owner__login',
    )

    select_related = ('author',)
    prefetch_related = (
        'control_test__tester',
        'control_test__evidence',
        'control_test__deficiency',
        'control_test__control_plan__process',
        'control_test__control_plan__system',
        'control_test__control_plan__service',
        'control_test__control_plan__risk',
        'control_test__control_plan__account',
        'control_test__control_plan__legal',
        'control_test__control_plan__business_unit',
        'control_test__control_plan__assertion',
        'control_test__control_plan__owner',
        'control_test__control_plan__reviewer',
        'control_test__control_plan__process__parent',
        'control_test__control_plan__account__parent',
        'control_test__controlstep_set',
        'control_test__control_plan',
        'control_test__reviewer',
        'control_test',
    )


class DeficiencyGroupListView(RelatedMixin, ListCreateAPIView):
    model = models.DeficiencyGroup
    serializer_class = serializers.DeficiencyGroupSerializer
    search_fields = (
        'full_description',
        'mitigating_factors',
        'comment',
    )
    select_related = (
        'author',
    )
    prefetch_related = (
        'deficiencies',
        'deficiencies__control_test',
        'deficiencies__control_test__tester',
        'deficiencies__control_test__evidence',
        'deficiencies__control_test__control_plan',
        'deficiencies__control_test__control_plan__process',
        'deficiencies__control_test__control_plan__system',
        'deficiencies__control_test__control_plan__service',
        'deficiencies__control_test__control_plan__risk',
        'deficiencies__control_test__control_plan__account',
        'deficiencies__control_test__control_plan__legal',
        'deficiencies__control_test__control_plan__business_unit',
        'deficiencies__control_test__control_plan__assertion',
        'deficiencies__control_test__control_plan__owner',
        'deficiencies__control_test__control_plan__reviewer',
        'deficiencies__control_test__control_plan__process__parent',
        'deficiencies__control_test__control_plan__account__parent',
        'deficiencies__control_test__controlstep_set',
        'deficiencies__control_test__reviewer',
    )


class IPEListView(RelatedMixin, ListCreateAPIView):
    model = models.IPE
    serializer_class = serializers.IPESerializer
    search_fields = (
        'name',
        'source_data',
        'report_logic',
        'system__name',
        'status',
    )
    select_related = ('author', 'system')
    prefetch_related = ('evidence', 'reviewer')


class LegalListView(ListCreateAPIView):
    model = models.Legal
    serializer_class = serializers.LegalSerializer
    search_fields = ('name',)


class RiskListView(ListCreateAPIView):
    model = models.Risk
    serializer_class = serializers.RiskSerializer
    search_fields = ('name', 'number')


class ServiceListView(RelatedMixin, ListCreateAPIView):
    model = models.Service
    serializer_class = serializers.ServiceSerializer
    search_fields = ('name', 'service_description')


class SystemListView(RelatedMixin, ListCreateAPIView):
    model = models.System
    serializer_class = serializers.SystemSerializer
    search_fields = ('name', 'service_description')


class ProcessListView(RelatedMixin, ListCreateAPIView):
    model = models.Process
    serializer_class = serializers.ProcessSerializer
    search_fields = ('name', 'process_type')
    select_related = ('parent',)


class FileListView(RelatedMixin, ListCreateAPIView):
    model = File
    serializer_class = serializers.FileDetailSerializer
    search_fields = ('name', 'content_type')

    def post(self, request, *args, **kwargs):
        file_data = get_first_file_from_request(request)
        self._check_file_size(file_data, settings.MAX_FILE_SIZE)
        return super().post(request, *args, **kwargs)

    def _check_file_size(self, file_data, max_file_size):
        if file_data.size > max_file_size:
            message = _('File size should be under: {}'.format(max_file_size))
            raise ValidationError(message)


class ControlTestIPEListView(RelatedMixin, ListCreateAPIView):
    model = models.ControlTestIPE
    serializer_class = serializers.ControlTestIPESerializer

    select_related = (
        'ipe',
        'control_test',
        'ipe__system',
        'ipe__author',
    )
    prefetch_related = ('attach', 'ipe__reviewer')
