import shlex

from django.utils.functional import cached_property
from django.core.exceptions import FieldError
from django.utils.translation import ugettext

from .. import errors
from ..logic.permissions import AuditBasePermission, AuditViewDependentPermission


class RelatedMixin:
    """
    Учитывает параметр fields в запросе
    не делает select/prefetch related запросы
    если эти поля не нужны в выдаче

    Следут ставить его первым в списке базовых классов
    для корректной работы
    """
    select_related = tuple()
    prefetch_related = tuple()
    
    def get_queryset(self):
        queryset = super().get_queryset()
        select_related, prefetch_related = self.get_related()
        if select_related:
            queryset = queryset.select_related(*select_related)
        if prefetch_related:
            queryset = queryset.prefetch_related(*prefetch_related)
        return queryset

    def get_related(self):
        fields = self.request.query_params.get('fields')
        select_related, prefetch_related = self.prepared_select_related, self.prepared_prefetch_related
        if fields is not None:
            fields = set(fields.split(','))
            select_related = (field for field in select_related if field.split('__')[0] in fields)
            prefetch_related = (field for field in prefetch_related if field.split('__')[0] in fields)
        return select_related, prefetch_related

    @cached_property
    def prepared_select_related(self):
        return self.select_related

    @cached_property
    def prepared_prefetch_related(self):
        return self.prefetch_related


class FilterMixin:
    """
    Учитывает параметр filter_by в запросе
    фильтрует queryset в соответствии с
    переданными значениями
    """
    def get_queryset(self):
        filter_by = self.request.query_params.get('filter_by')
        query = {}
        if filter_by:
            query = self.get_parsed_query(filter_by)
        try:
            return self.model.objects.filter(**query)
        except FieldError:
            # Translators: Используется в ответе апи в случае если был сделан запрос с некорректным параметром
            raise errors.BadRequestError(
                ugettext('You have to re-create the filter, your version of the filter is deprecated')
            )

    def get_parsed_query(self, query):
        """
        Получаем из строки запроса словарь
        вида {параметр: условие}. Если параметр
        оканчивается на __gte/__lte, возвращаем строку,
        в остальных случаях возвращаем список.

        :param query: str
        :rtype: dict(str: list) or dict(str: str)
        """
        parsed_query = {}

        request_args = shlex.split(query)
        for query in request_args:
            param, term = query.split(':', 1)
            if param[-5:] in ('__gte', '__lte'):
                parsed_query[param] = term
                continue
            format_name = '{}__{}'.format(param, 'in')
            result_term = term.split(',')
            parsed_query[format_name] = result_term

        return parsed_query

    def remove_duplicate(self, queryset):
        """
        Следует применять этот метод на queryset
        перед возвратом пользователю ответа, так как
        orm система django может порождать дубликаты
        при фильтрации по смежным сущностям
        """
        duplicate = set()
        duplicate_add = duplicate.add
        return [obj for obj in queryset if not (obj in duplicate or duplicate_add(obj))]


class BasePermissionsMixin:
    permission_classes = AuditBasePermission,


class ViewDependentPermissionsMixin:
    permission_classes = AuditViewDependentPermission,

    def get_perm_to_check(self, request):
        raise NotImplementedError(
            '{cls}.get_perm_to_check() must be implemented.'.format(
                cls=self.__class__.__name__
            )
        )


class SendToDraftMixin:

    def perform_update(self, serializer):
        obj = self.get_object()
        additional_kwargs = {}
        statuses = self.STATUSES
        if (
                obj.status in (statuses.review, statuses.active) and
                not obj.reviewer.filter(uid=self.request.user.uid).exists()
        ):
            additional_kwargs['status'] = statuses.draft
        serializer.save(**additional_kwargs)
