import collections.abc
import json
import logging

from django.conf import settings
from django.http import Http404
from django_tools_log_context import request_context, request_profiler
from rest_framework import filters, generics
from rest_framework.authentication import BaseAuthentication
from rest_framework.exceptions import APIException
from rest_framework.response import Response

from intranet.audit.src.api_v1 import errors
from intranet.audit.src.api_v1.ordering import RelatedOrderingFilter
from intranet.audit.src.api_v1.views.metadata import AuditMetadata
from intranet.audit.src.api_v1.views.mixins import BasePermissionsMixin, FilterMixin


logger = logging.getLogger(__file__)


class YauthAuthentication(BaseAuthentication):

    def authenticate(self, request):
        if request.yauser and request.yauser.is_authenticated():
            return request._request.user, None
        else:
            raise errors.YauthError


class APIView(generics.GenericAPIView):
    """
    Базовый класс для всех API View
    """
    authentication_classes = (YauthAuthentication,)
    metadata_class = AuditMetadata

    def dispatch(self, request, *args, **kwargs):
        with request_context(request, endpoint=self):
            with request_profiler(request, threshold=settings.THRESHOLD):
                response = super().dispatch(request, *args, **kwargs)

        return response

    def get_data_from_request(self, *path, data=None, index=0, required=True, default=None):
        data = data or self.request.data
        value = data.get(path[index])
        if not value and required:
            raise errors.BadRequestError('You should specify {} in request'.format(path[index]))
        if path[index] not in data:
            return default
        index += 1
        if len(path) == index:
            return value
        return self.get_data_from_request(*path, data=value, index=index)

    def handle_exception(self, exc):
        """
        Обработать исключения из хэндлера запроса
        @param exc: объект исключения
        """
        if isinstance(exc, errors.RestApiError):
            data = errors.rest_api_error_to_dict(exc)
            response = Response(data, status=exc.status_code, exception=True)

        elif isinstance(exc, APIException):
            if isinstance(exc.detail, dict):
                exc_data = exc.detail
            else:
                if isinstance(exc.detail, (list, tuple)):
                    detail = exc.detail
                else:
                    detail = [exc.detail]
                exc_data = {'detail': detail}

            data = {
                'error_code': errors.UnhandledException.error_code,
                'level': 'ERROR',
                'message': [errors.get_error_message(exc)],
                'debug_message': exc.default_detail,
                'errors': exc_data,
            }
            response = Response(data, status=exc.status_code, exception=True)
        else:
            logger.exception('Error happened while handling request "%s"', repr(exc))
            # 500я ошибка
            data = {
                'error_code': errors.UnhandledException.error_code,
                'level': 'ERROR',
                'debug_message': '{0}: {1}'.format(
                    errors.UnhandledException.debug_message, repr(','.join(exc.args))),
                'message': errors.UnhandledException().non_field_messages,
            }
            response = Response(data, status=500, exception=True)
        logger.info('Responded with "%s"', response.status_code)
        return response

    def successful_response(self):
        return Response(status=200)

    def get_param_from_request(self, request, param_name):
        """
        Получаем значение параметра либо из body запроса или
        из request.POST - это нужно, чтобы работали запросы
        через django формы, а так же запросы через curl/requests/...
        """
        # TODO: поменять на request.data везде в коде, когда откажемся от action через админку
        param = None
        try:
            param = json.loads(request.body.decode()).get(param_name)
        except json.JSONDecodeError:
            pass
        if not param:
            param = request.POST.get(param_name)
        return param

    def permission_denied(self, request, message=None):
        if request.authenticators and not request.successful_authenticator:
            raise errors.YauthError()

        raise errors.AccessError(message)

    def get_serializer_context(self):
        context = super(APIView, self).get_serializer_context()
        context['user'] = self.request.user
        return context


class ListCreateAPIView(BasePermissionsMixin, FilterMixin, APIView, generics.ListCreateAPIView):

    filter_backends = filters.SearchFilter, RelatedOrderingFilter,
    ordering = '-id',

    def get_serializer(self, *args, **kwargs):
        # Убираем дубликаты, так как работа orm django не гарантирует
        # их отсутствие при фильтрации по many-to-many, чтобы везде
        # distinct не вызывать, так как функция ожидает iterable
        # проверяем тип перед фильтрацией
        if len(args) == 1 and isinstance(args[0], collections.abc.Iterable):
            queryset = self.remove_duplicate(args[0])
            args = queryset,
        return super().get_serializer(*args, **kwargs)

    def list(self, request, *args, **kwargs):
        response = super().list(request, args, kwargs)
        user = request.user
        response.data['actions'] = {
            'add': user.has_perm('{}.add_{}'.format(
                self.model._meta.app_label,
                self.model.__name__.lower(),
            )),
        }
        return response


class RetrieveUpdateAPIView(BasePermissionsMixin, generics.RetrieveUpdateDestroyAPIView, APIView):

    def get_object(self):
        try:
            return super().get_object()
        except Http404:
            raise errors.NotFoundError()


class ListAPIView(BasePermissionsMixin, FilterMixin, APIView, generics.ListAPIView):

    filter_backends = filters.SearchFilter, RelatedOrderingFilter,
    ordering = '-id',
