import collections

from django.forms import Textarea
from django import db
from django.contrib import admin, messages
from django.core.urlresolvers import reverse
from django.utils.html import format_html
from django.contrib.admin.helpers import ActionForm
from django.utils.translation import ugettext
from django.core import urlresolvers
from django.utils.safestring import mark_safe
from django.http import HttpResponseRedirect
from django.conf import settings
from django.contrib.auth.models import Permission

from ajax_select import make_ajax_form

from ajax_select.admin import AjaxSelectAdmin, AjaxSelectAdminTabularInline
from ajax_select.fields import AutoCompleteSelectMultipleField

from advanced_filters.admin import AdminAdvancedFiltersMixin

from . import models
from .logic.form import update_form_data, prepare_form_data
from intranet.audit.src.api_v1.views.controlplan import RunControlPlanView
from intranet.audit.src.users.dao.user import get_stated_persons_by_uids_from_staff
from intranet.audit.src.users.logic.user import already_created

DELIMITER = ',<br>'


class AssigneeChoice(ActionForm):
    assign_to = AutoCompleteSelectMultipleField('stated_person', required=False, help_text=None)


class ControlStepInline(AjaxSelectAdminTabularInline):
    extra = 0
    model = models.ControlStep
    form = make_ajax_form(models.ControlStep, {
        'file': 'file',
    })
    fields = 'step', 'result', 'comment', 'file',
    formfield_overrides = {
        db.models.TextField: {'widget': Textarea(attrs={'rows': 2, 'cols': 60})},
    }


# Translators: название сервиса, отображается в шапке админки
admin.site.site_header = ugettext('Я.Аудит')
admin.site.site_title = ugettext('Я.Аудит')
admin.site.index_title = ''


class RootProcessListFilter(admin.SimpleListFilter):
    # Translators: Название блоки фильтрации в админке для сущности Process
    title = ugettext('Process')
    parameter_name = 'process__id'
    model = models.Process

    def lookups(self, request, model_admin):
        return [(process.id, process) for process in
                self.model.objects.filter(process_type=self.model.TYPES.root)]

    def queryset(self, request, queryset):
        if not self.value():
            return queryset
        return queryset.filter(process__id=self.value())


class BaseObjectAdmin(AjaxSelectAdmin, admin.ModelAdmin):
    search_fields = 'id', 'created',
    save_as = True
    list_display = 'creation_date',
    exclude = 'is_removed',
    readonly_fields = 'author', 'created', 'modified',
    list_per_page = 25

    def save_model(self, request, obj, form, change):
        if getattr(obj, 'author', None) is None:
            obj.author = request.user
        obj.save()

    def save_formset(self, request, form, formset, change):
        for f in formset.forms:
            if getattr(f.instance, 'author', None) is None:
                f.instance.author = request.user
        formset.save()

    def format_item_display(self, item):
        link = urlresolvers.reverse("admin:core_{}_change".format(item.__class__.__name__.lower(),
                                                                  ),
                                    args=[item.id])
        return mark_safe(u'<a href="%s">%s</a>' % (link, item))

    formfield_overrides = {
        db.models.TextField: {'widget': Textarea(attrs={'rows': 2, 'cols': 90})},
    }

    def get_actions(self, request):
        actions = super().get_actions(request)
        perm = '{}.{}_{}'.format(self.opts.app_label, 'delete', self.model.__name__.lower())
        if not request.user.has_perm(perm):
            if 'delete_selected' in actions:
                del actions['delete_selected']
        return actions

    def lookup_allowed(self, lookup, value):
        return True


class TestPeriodDisplayMixin:

    class TestPeriodHelper:
        """
        Используется чтобы формировать
        различные названия колонок для
        отображения данных в
        классах наследующих текущий
        """
        def __init__(self, field, slug):
            self.field = field
            self.slug = slug

        def __call__(self, obj):
            date = getattr(obj, self.field)
            if date:
                return date.strftime(settings.DATE_DISPLAY_FORMAT)
            return obj.NO_DATE

        @property
        def short_description(self):
            period_value = self.field.split('_')[-1]
            return '{} period {}'.format(self.slug,
                                         period_value,
                                         )

        @property
        def admin_order_field(self):
            return self.field


class SearchControlTestMixin:
    search_fields = ('controltest__control_plan__process__name',
                     'controltest__control_plan__service__name',
                     'controltest__control_plan__system__name',
                     'controltest__control_plan__risk__name',
                     'controltest__control_plan__control__name',
                     'controltest__control_plan__account__name',
                     'controltest__control_plan__legal__name',
                     'controltest__control_plan__business_unit__name',
                     'controltest__control_plan__assertion__name',
                     'controltest__control_plan__control__number',
                     'controltest__control_plan__risk__number',
                     'controltest__control_plan__owner__last_name',
                     'controltest__control_plan__owner__first_name',
                     'controltest__control_plan__owner__login',
                     )


class ExportActionMixin:
    actions = ['export', ]

    def export(self, request, queryset):
        obj_class = queryset.first().__class__.__name__.lower()
        api_url = urlresolvers.reverse("api_v1:export",
                                       kwargs={'obj_class': obj_class,
                                               },
                                       )
        query = 'obj_pks={}'.format(','.join(str(obj.id) for obj in queryset))
        final_url = '{}?{}'.format(api_url, query)
        return HttpResponseRedirect(final_url)


class StatedPersonMixin:
    PERSONS_PREFETCH = ('owner', 'responsible',)
    PERSONS_ATTRIBUTE_NAMES = ('owner', 'responsible', )

    class StatedPersonHelper:
        def __init__(self, field):
            self.field = field

        def __call__(self, obj):
            persons = getattr(obj, self.field)
            return mark_safe(DELIMITER.join(person.format_item_display
                                            for person in persons.all()))

        @property
        def short_description(self):
            return self.field

        @property
        def admin_order_field(self):
            return self.field

    def get_queryset(self, request):
        query = super().get_queryset(request)
        query = query.prefetch_related(*self.PERSONS_PREFETCH)
        return query

    def save_model(self, request, obj, form, change):
        cleaned_form = form.cleaned_data

        ids_data = collections.defaultdict(set)
        for attr_name in self.PERSONS_ATTRIBUTE_NAMES:
            for uid in cleaned_form.get(attr_name, []):
                if not already_created(uid):
                    ids_data[uid].add(attr_name)

        ids = [uid for uid in ids_data]
        if ids:
            users = get_stated_persons_by_uids_from_staff(*ids,
                                                          request=request,
                                                          )
            prepare_form_data(self.PERSONS_ATTRIBUTE_NAMES,
                              cleaned_form,
                              )
            update_form_data(ids_data,
                             users,
                             cleaned_form,
                             )
        super().save_model(request, obj, form, change)


class NamedObjectAdmin(BaseObjectAdmin):
    search_fields = BaseObjectAdmin.search_fields + ('name', )
    list_display = ('name', )


class ControlDisplayAdmin(BaseObjectAdmin):
    readonly_fields = ('show_controls', ) + BaseObjectAdmin.readonly_fields
    list_display = ('show_controls', ) + BaseObjectAdmin.list_display

    def show_controls(self, obj):
        if obj.pk:
            controls = set(control_test.control_plan.control for control_test
                           in obj.controltest_set.select_related('control_plan__control'))
            if controls:
                return format_html('<br>'.join(
                    self.format_item_display(control) for control
                    in controls
                ))
            # Translators: Выводится в админке если у сущности еще нет привязанных объектов Control
            return ugettext('No controls attached')
        # Translators: Выводится в админке если объект еще не сохранен
        return ugettext('Save object first')

    # Translators: Наименование поля в админке для отображения привязанных объектов Control
    show_controls.short_description = ugettext('Controls')
    show_controls.allow_tags = True


class SystemObjectAdmin(NamedObjectAdmin):
    list_display = 'name', 'service_description',

    form = make_ajax_form(models.System, {
        'service_description': 'abc_service',
    })


class ServiceObjectAdmin(NamedObjectAdmin):
    list_display = ('name', 'service_description', )

    form = make_ajax_form(models.Service, {
        'service_description': 'abc_service',
    })


class ProcessAdmin(NamedObjectAdmin):
    list_select_related = ('parent', )
    list_display = ('name', 'process_type', 'parent', )
    form = make_ajax_form(models.Process, {
        'parent': 'process_no_add',
    })


class LegalAdmin(NamedObjectAdmin):
    pass


class AccountAdmin(NamedObjectAdmin):
    list_select_related = ('parent',)
    list_display = ('name', 'parent')


class BusinessUnitAdmin(NamedObjectAdmin):
    pass


class IPEAdmin(StatedPersonMixin, SearchControlTestMixin,
               AdminAdvancedFiltersMixin, ExportActionMixin,
               BaseObjectAdmin, ):
    PERSONS_ATTRIBUTE_NAMES = 'reviewer',
    PERSONS_PREFETCH = 'reviewer',
    search_fields = (BaseObjectAdmin.search_fields +
                     ('full_description', 'name',)
                     )

    list_display = ('name', 'appliance', 'ipe_type', 'status', )

    form = make_ajax_form(models.IPE, {
        'evidence': 'file',
        'service_description': 'abc_service',
        'system': 'system',
        'reviewer': 'stated_person',
    })

    advanced_filter_fields = (
        # Translators: Название поля appliance для отображения при построении фильтра в админке для модели IPE
        ('appliance', ugettext('appliance')),
        # Translators: Название поля service_description для отображения при построении фильтра в админке для модели IPE
        ('service_description', ugettext('service description')),

    )


class AssertionAdmin(NamedObjectAdmin):
    pass


class ControlPlanAdmin(StatedPersonMixin, AdminAdvancedFiltersMixin, TestPeriodDisplayMixin,
                       ExportActionMixin, BaseObjectAdmin):
    PERSONS_PREFETCH = 'owner', 'reviewer',
    PERSONS_ATTRIBUTE_NAMES = 'owner', 'reviewer',
    search_fields = BaseObjectAdmin.search_fields + ('description', 'process__name',
                                                     'service__name', 'system__name',
                                                     'risk__name', 'control__name',
                                                     'account__name', 'legal__name',
                                                     'business_unit__name',
                                                     'assertion__name', 'owner__login',
                                                     'control__number', 'risk__number',
                                                     'owner__last_name', 'owner__first_name',
                                                     )
    action_form = AssigneeChoice

    readonly_fields = ('create_control_test', 'control_tests_attached', ) + BaseObjectAdmin.readonly_fields
    list_filter = RootProcessListFilter, 'system', 'service', 'risk',
    list_select_related = ('control',)

    list_display = ('control_display', 'control_name', 'risks_link',
                    'test_period_started_display', 'test_period_finished_display',
                    'process_link',
                    'services_link', 'system_link',
                    'business_unit_link', 'legals_link',
                    )

    test_period_started_display = TestPeriodDisplayMixin.TestPeriodHelper('test_period_started',
                                                                          'Plan',
                                                                          )

    test_period_finished_display = TestPeriodDisplayMixin.TestPeriodHelper('test_period_finished',
                                                                           'Plan',
                                                                           )

    def control_display(self, obj):
        control = obj.control
        if control:
            return control.number

    control_display.admin_order_field = 'control'
    # Translators: Выводится в админке ControlPlan как название колонки для отображения
    # number у связанных Control
    control_display.short_description = ugettext('Control')

    def control_name(self, obj):
        control = obj.control
        if control:
            return control.name

    control_name.admin_order_field = 'control'
    # Translators: Выводится в админке ControlPlan как название колонки для
    # отображения name у связанных Control
    control_name.short_description = ugettext('Control name')

    def risks_link(self, obj):
        return mark_safe(DELIMITER.join(risk.number for risk in obj.risk.all()))

    risks_link.admin_order_field = 'risk'
    # Translators: Выводится в админке ControlPlan как название колонки
    # для отображения связанных Risk
    risks_link.short_description = ugettext('Risk')

    def process_link(self, obj):
        return mark_safe(DELIMITER.join(process.name for process in obj.process.all()))
    process_link.admin_order_field = 'process'
    # Translators: Выводится в админке ControlPlan как название колонки
    # для отображения связанных Process
    process_link.short_description = ugettext('Process')

    def system_link(self, obj):
        return mark_safe(DELIMITER.join(system.name for system in obj.system.all()))
    system_link.admin_order_field = 'system'
    # Translators: Выводится в админке ControlPlan как название колонки
    # для отображения связанных System
    system_link.short_description = ugettext('System')

    def services_link(self, obj):
        return mark_safe(DELIMITER.join(service.name for service in obj.service.all()))
    services_link.admin_order_field = 'service'
    # Translators: Выводится в админке ControlPlan как название колонки
    # для отображения связанных Service
    services_link.short_description = ugettext('Service')

    def legals_link(self, obj):
        return mark_safe(DELIMITER.join(legal.name for legal in obj.legal.all()))
    legals_link.admin_order_field = 'legal'
    # Translators: Выводится в админке ControlPlan как название колонки
    # для отображения связанных Legal
    legals_link.short_description = ugettext('Legal')

    def business_unit_link(self, obj):
        return mark_safe(DELIMITER.join(unit.name for unit in obj.business_unit.all()))
    business_unit_link.admin_order_field = 'business_unit'
    # Translators: Выводится в админке ControlPlan как название колонки
    # для отображения связанных business_unit
    business_unit_link.short_description = ugettext('BU')

    def get_queryset(self, request):
        query = super().get_queryset(request)
        query = query.prefetch_related('service', 'system', 'risk',
                                       'legal',
                                       'business_unit', 'process',
                                       )
        return query

    form = make_ajax_form(models.ControlPlan, {
        'service': 'service',
        'system': 'system',
        'process': 'process',
        'risk': 'risk',
        'control': 'control',
        'account': 'account',
        'legal': 'legal',
        'business_unit': 'business_unit',
        'owner': 'stated_person',
        'reviewer': 'stated_person',

    })

    advanced_filter_fields = (
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlPlan по name связанного Control
        ('control__name', ugettext('control name')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlPlan по name связанного Risk
        ('risk__name', ugettext('risks name')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlPlan по name связанного Process
        ('process__name', ugettext('process name')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlPlan по name связанного System
        ('system__name', ugettext('system name')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlPlan по name связанного Service
        ('service__name', ugettext('service name')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlPlan по name связанного Legal
        ('legal__name', ugettext('legal name')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlPlan по name связанного BU
        ('business_unit__name', ugettext('business unit name')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlPlan по name связанного Assertion
        ('assertion__name', ugettext('assertion name')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlPlan по значению control_type
        ('control_type', ugettext('control type')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlPlan по значению frequency
        ('frequency', ugettext('frequency')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlPlan по значению method
        ('method', ugettext('method')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlPlan по значению key_control
        ('key_control', ugettext('key control')),
    )

    actions = ['create_control_tests', ] + ExportActionMixin.actions

    def create_control_tests(self, request, queryset):
        request.POST = request.POST.copy()
        request.POST['obj_pks'] = [obj.id for obj in queryset]
        assign_to = request.POST['assign_to']
        if assign_to == '|':
            del request.POST['assign_to']
        else:
            request.POST['assign_to'] = ','.join(uid for uid
                                                 in assign_to.strip('|').split('|'))
        RunControlPlanView().post(request)
        messages.success(request, 'Create tests for selected control plans')

    def get_actions(self, request):
        actions = super().get_actions(request)
        if not request.user.has_perm('core.add_controltest'):
            if 'create_control_tests' in actions:
                del actions['create_control_tests']
        return actions

    def get_readonly_fields(self, request, obj=None):
        readonly_fields = super().get_readonly_fields(request, obj)
        if not request.user.has_perm('core.add_controltest'):
            readonly_fields = tuple(field for field
                                    in self.readonly_fields if field != 'run_plan')
        return readonly_fields

    def create_control_test(self, obj):
        if obj.pk:
            return format_html(
                '''
                    <input id="obj_pks_id"
                    type="hidden" name="obj_pks"
                    value="{}"></input>

                    <button type="submit"
                    formmethod="post"
                    class="btn btn-primary"
                    formaction="{}">
                    Create control test
                    </button>
                ''',
                obj.pk,
                reverse('api_v1:controlplan_run'),
            )
        # Translators: Используется в админке ControlPlan в качестве заглушки для действия
        # если объект еще не сохранен
        return ugettext('Save object first')

    # Translators: Используется в админке ControlPlan в качестве названия поля для запуска ControlPlan
    create_control_test.short_description = ugettext('Action')
    create_control_test.allow_tags = True

    def control_tests_attached(self, obj):
        if obj.pk:
            control_tests = obj.controltest_set.all()
            if control_tests:
                return format_html('<br>'.join(
                    self.format_item_display(control_test) for control_test
                    in control_tests
                ))
            # Translators: Выводится в админке ControlPlan, если к ControlPlan еще н
            # е прикреплены ControlTest
            return ugettext('No control tests attached')
        # Translators: Используется в админке ControlPlan в качестве заглушки для
        # действия если объект еще не сохранен
        return ugettext('Save object first')
    # Translators: Используется в админке ControlPlan в качестве названия поля для
    # отображения списка связанных ControlTest
    control_tests_attached.short_description = ugettext('Control Test')
    control_tests_attached.allow_tags = True

    def get_form(self, request, obj=None, **kwargs):
        form = super().get_form(request, obj, **kwargs)
        form.base_fields['assertion'].widget.attrs['style'] = 'height: 140px;'
        return form


class ControlTestAdmin(StatedPersonMixin, TestPeriodDisplayMixin, AdminAdvancedFiltersMixin,
                       ExportActionMixin, BaseObjectAdmin):
    PERSONS_PREFETCH = ('tester', 'reviewer', )
    PERSONS_ATTRIBUTE_NAMES = ('tester', 'reviewer', )
    readonly_fields = BaseObjectAdmin.readonly_fields + ('controlstep_order', )

    list_filter = 'control_plan__risk', 'control_plan__control'
    list_select_related = ('control_plan', )
    list_display = ('control_number', 'control_name', 'control_plan',
                    'test_period_started_display', 'test_period_finished_display',
                    'design_efficiency', 'operational_efficiency', 'deficiency',
                    'status',
                    )

    search_fields = ('tester__login',
                     'tester__first_name',
                     'tester__last_name',
                     'test_period_started',
                     'test_period_finished',
                     'testing_date',
                     'sampling',
                     'evidence_comments',
                     'deficiency__short_description',
                     'deficiency__full_description',
                     'deficiency__ticket_key',
                     'deficiency__mitigating_factors',
                     'control_plan__process__name',
                     'control_plan__service__name',
                     'control_plan__system__name',
                     'control_plan__risk__name',
                     'control_plan__control__name',
                     'control_plan__account__name',
                     'control_plan__legal__name',
                     'control_plan__business_unit__name',
                     'control_plan__assertion__name',
                     'control_plan__control__number',
                     'control_plan__risk__number',
                     'control_plan__owner__last_name',
                     'control_plan__owner__first_name',
                     'control_plan__owner__login',
                     'controlstep__step',
                     )

    form = make_ajax_form(models.ControlTest, {
        'control_plan': 'control_plan',
        'tester': 'stated_person',
        'evidence': 'file',
        'deficiency': 'deficiency',
        'reviewer': 'stated_person',
    })

    inlines = ControlStepInline,

    test_period_started_display = TestPeriodDisplayMixin.TestPeriodHelper('test_period_started',
                                                                          'Test',
                                                                          )

    test_period_finished_display = TestPeriodDisplayMixin.TestPeriodHelper('test_period_finished',
                                                                           'Test',
                                                                           )

    def controlstep_order(self, obj):
        return list(obj.get_controlstep_order())
    # Translators: Используется в админке ControlPlan в качестве названия поля
    # для отображения порядка связанных ControlStep
    controlstep_order.short_description = ugettext('Control Step order')

    def control_number(self, obj):
        if obj.control_plan:
            return obj.control_plan.control.number
    control_number.admin_order_field = 'control_plan__control__number'
    # Translators: Используется в админке ControlPlan в качестве названия колонки
    # для отображения number у связанных Control
    control_number.short_description = ugettext('Control')

    def control_name(self, obj):
        if obj.control_plan:
            return obj.control_plan.control.name

    control_name.admin_order_field = 'control_plan__control__name'
    # Translators: Используется в админке ControlPlan в качестве названия колонки
    # для отображения name у связанных Control
    control_name.short_description = ugettext('Control name')

    def deficiency(self, obj):
        return format_html('<br>'.join(
            self.format_item_display(deficiency) for deficiency
            in obj.deficiency.all()
        ))
    deficiency.admin_order_field = 'deficiency'
    # Translators: Используется в админке ControlPlan в качестве названия колонки
    # для отображения связанных Deficiency
    deficiency.short_description = ugettext('Deficiency')

    advanced_filter_fields = (
        # Translators: Название поля для отображения при построении фильтра в
        # админке для модели ControlTest по login связанного Tester
        ('tester__login', ugettext('tester')),
        # Translators: Название поля для отображения при построении фильтра в
        # админке для модели ControlTest по значению test_period_started
        ('test_period_started', ugettext('test_period_started')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по значению test_period_finished
        ('test_period_finished', ugettext('test_period_finished')),

        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по name связанного Control
        ('control_plan__control__name', ugettext('control name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по name связанного Risk
        ('control_plan__risk__name', ugettext('risks name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по name связанного Process
        ('control_plan__process__name', ugettext('process name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по name связанного System
        ('control_plan__system__name', ugettext('system name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по name связанного Service
        ('control_plan__service__name', ugettext('service name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по name связанного Legal
        ('control_plan__legal__name', ugettext('legal name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по name связанного BU
        ('control_plan__business_unit__name', ugettext('business unit name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по name связанного Assertion
        ('control_plan__assertion__name', ugettext('assertion name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по control_type связанного ControlPlan
        ('control_plan__control_type', ugettext('control type (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по frequency связанного ControlPlan
        ('control_plan__frequency', ugettext('frequency (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по method связанного ControlPlan
        ('control_plan__method', ugettext('method (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели ControlTest по key_control связанного ControlPlan
        ('control_plan__key_control', ugettext('key control (control plan)')),
    )


class RiskAdmin(NamedObjectAdmin):
    search_fields = NamedObjectAdmin.search_fields + ('number', )
    list_display = ('number', 'name', )


class ControlAdmin(NamedObjectAdmin):
    search_fields = ('number', 'name', )
    list_display = ('number', 'name', )
    list_filter = 'controlplan__process',


class ControlStepAdmin(BaseObjectAdmin):
    list_display = ('step', 'result', ) + BaseObjectAdmin.list_display
    search_fields = ('step', ) + BaseObjectAdmin.search_fields
    list_filter = 'control_test__control_plan__process',

    form = make_ajax_form(models.ControlStep, {
        'control_test': 'control_test',
        'file': 'file',
    })


class DeficiencyAdmin(SearchControlTestMixin, AdminAdvancedFiltersMixin,
                      ExportActionMixin, BaseObjectAdmin, ):
    search_fields = (SearchControlTestMixin.search_fields +
                     ('short_description', 'full_description',)
                     )
    list_display = ('control_test_attached', 'short_description',
                    'significance_evaluation', 'state',
                    )

    def control_test_attached(self, obj):
        control_tests = obj.control_test.all()
        if not control_tests:
            return 'No control tests attached'
        return format_html('<br>'.join(str(control_test) for control_test in
                                       control_tests))

    # Translators: Используется в админке Deficiency в качестве названия колонки
    # для отображения связанных ControlTest
    control_test_attached.short_description = ugettext('Control Test')

    advanced_filter_fields = (
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по значению state
        ('state', ugettext('state')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по значению ticket_key
        ('ticket_key', ugettext('ticket key')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по значению potential_impact
        ('potential_impact', ugettext('potential impact')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по значению misstatement_probability
        ('misstatement_probability', ugettext('misstatement probability')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по значению significance_evaluation
        ('significance_evaluation', ugettext('significance evaluation')),

        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по tester связанного ControlTest
        ('control_test__tester__login', ugettext('tester (control test)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по test_period_started связанного ControlTest
        ('control_test__test_period_started', ugettext('test_period_started (control test)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по test_period_finished связанного ControlTest
        ('control_test__test_period_finished', ugettext('test_period_finished (control test)')),

        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по name связанного Control
        ('control_test__control_plan__control__name', ugettext('control name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по name связанного Risk
        ('control_test__control_plan__risk__name', ugettext('risks name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по name связанного Process
        ('control_test__control_plan__process__name', ugettext('process name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по name связанного System
        ('control_test__control_plan__system__name', ugettext('system name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по name связанного Service
        ('control_test__control_plan__service__name', ugettext('service name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по name связанного Legal
        ('control_test__control_plan__legal__name', ugettext('legal name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по name связанного BU
        ('control_test__control_plan__business_unit__name', ugettext('business unit name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по name связанного Assertion
        ('control_test__control_plan__assertion__name', ugettext('assertion name (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по control_type связанного ControlPlan
        ('control_test__control_plan__control_type', ugettext('control type (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по frequency связанного ControlPlan
        ('control_test__control_plan__frequency', ugettext('frequency (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по method связанного ControlPlan
        ('control_test__control_plan__method', ugettext('method (control plan)')),
        # Translators: Название поля для отображения при построении фильтра в админке
        # для модели Deficiency по key_control связанного ControlPlan
        ('control_test__control_plan__key_control', ugettext('key control (control plan)')),
    )


class DeficiencyGroupAdmin(BaseObjectAdmin):

    list_display = (
        'id',
        'state',
        'full_description',
        'created',
    )

    form = make_ajax_form(
        model=models.DeficiencyGroup,
        fieldlist={
            'deficiencies': 'deficiency',
        },
    )


class ControlTestIPEAdmin(BaseObjectAdmin):
    form = make_ajax_form(models.ControlTestIPE, {
        'ipe': 'ipe',
        'control_test': 'control_test',
    })


admin.site.register(models.System, SystemObjectAdmin)
admin.site.register(models.Service, ServiceObjectAdmin)
admin.site.register(models.Process, ProcessAdmin)

admin.site.register(models.Legal, LegalAdmin)
admin.site.register(models.Account, AccountAdmin)
admin.site.register(models.BusinessUnit, BusinessUnitAdmin)
admin.site.register(models.Assertion, AssertionAdmin)

admin.site.register(models.Control, ControlAdmin)
admin.site.register(models.ControlPlan, ControlPlanAdmin)
admin.site.register(models.ControlTest, ControlTestAdmin)
admin.site.register(models.ControlStep, ControlStepAdmin)
admin.site.register(models.Deficiency, DeficiencyAdmin)
admin.site.register(models.DeficiencyGroup, DeficiencyGroupAdmin)

admin.site.register(models.IPE, IPEAdmin)
admin.site.register(models.Risk, RiskAdmin)
admin.site.register(models.ControlTestIPE, ControlTestIPEAdmin)
admin.site.register(Permission)
