from .. import models
from intranet.audit.src.core.logic.controltest import get_last_controltests_by_plans
from .controlsteps import copy_control_steps


def get_controltests_by_plans(*control_plans):
    return (
        models.ControlTest.objects
        .filter(control_plan__in=control_plans)
        .prefetch_related('controlstep_set')
    )


def create_controltests_by_plans(*control_plans, author, testers=None):
    """
    Создает новые control_test по переданным control_plans,
    копирует шаги с последнего control_test относящегося к
    определенному control_plan, если такой имеется
    """
    last_controltests = get_last_controltests_by_plans(*control_plans)

    new_controltests = (
        models.ControlTest(
            control_plan=control_plan,
            author=author,
        ) for control_plan in control_plans
    )

    new_controltests = {control_test.control_plan_id: control_test
                        for control_test in models.ControlTest.objects.bulk_create(new_controltests)}

    if testers:
        add_testers_to_control_tests(new_controltests.values(), testers)

    copy_control_steps(*last_controltests,
                       author=author,
                       new_control_tests=new_controltests,
                       )

    return new_controltests


def add_testers_to_control_tests(control_tests, testers):
    """
    Создает связи между StatedPerson и ControlTest,
    чтобы сделать это за один запрос к базе обращается
    напрямую к таблице с many-to-many связями между
    сущностями
    """
    through_model = models.ControlTest.tester.through
    relations_to_create = [
        through_model(controltest_id=control_test.pk, statedperson_id=tester.pk)
        for control_test in control_tests
        for tester in testers
    ]
    through_model.objects.bulk_create(relations_to_create)


def copy_controltest(control_test, extra_data, author_id):
    steps = control_test.controlstep_set.all()
    evidence = (obj.id for obj in control_test.evidence.all())

    control_test.id = None
    control_test.author_id = author_id
    control_test.test_period_started = extra_data['test_period_started']
    control_test.test_period_finished = extra_data['test_period_finished']
    control_test.status = models.ControlTest.STATUSES.draft
    control_test.save()
    control_test.evidence.add(*evidence)
    control_test.tester.add(*extra_data['tester'])
    new_steps = [
        models.ControlStep(
            control_test=control_test,
            step=step.step,
            author_id=control_test.author_id,
            _order=step._order,
            comment=step.comment,
        )
        for step in steps
    ]
    models.ControlStep.objects.bulk_create(new_steps)
    return control_test
