from functools import reduce
import operator

from django.db.models import Q
from django.core import urlresolvers

from ajax_select import register, LookupChannel

from . import models
from .logic.abc_service import get_abc_services


class BaseLookup(LookupChannel):
    lookup_fields = 'id__contains',
    wildcard_order = 'created'
    application = 'core'
    obj_pk_name = 'id'
    select_related = tuple()
    prefetch_related = tuple()

    def get_query(self, q, request):
        base_lookup_fields = getattr(self, 'base_lookup_fields', None)
        if base_lookup_fields:
            base_objects = self.model.objects.filter(**base_lookup_fields)
        else:
            base_objects = self.model.objects.all()

        base_objects = self.get_related_objects(base_objects)

        if q == '*':
            return self.get_wildcard_result(base_objects)

        query_objects = self.get_extended_query(q, self.lookup_fields)
        extended_query = reduce(operator.or_, query_objects)
        return set(
            base_objects
            .filter(extended_query)
            .order_by(self.wildcard_order)
        )

    def format_item_display(self, item):
        model_name = item.__class__.__name__.lower()
        view_name = f'admin:{self.application}_{model_name}_change'
        link = urlresolvers.reverse(view_name, args=[getattr(item, self.obj_pk_name)])
        return f'<a href="{link}">{item}</a>'

    def get_extended_query(self, q, fields):
        params = {field: q for field in fields}
        return self.get_query_objects(params)

    def get_query_objects(self, params):
        return (Q(lookup) for lookup in params.items())

    def get_wildcard_result(self, objects):
        return objects.order_by(self.wildcard_order)

    def get_related_objects(self, objects):
        if self.select_related:
            objects = objects.select_related(*self.select_related)
        if self.prefetch_related:
            objects = objects.prefetch_related(*self.prefetch_related)
        return objects


class NameLookup(BaseLookup):
    lookup_fields = BaseLookup.lookup_fields + ('name__icontains', )


class DescriptionLookup(BaseLookup):
    lookup_fields = BaseLookup.lookup_fields + ('description__icontains', )


class ProcessLookup(NameLookup):
    model = models.Process


@register('process_no_add')
class ProcessNoAdd(ProcessLookup):
    def can_add(self, user, model):
        return False


@register('root_process')
class RootProcessLookup(NameLookup):
    model = models.Process
    base_lookup_fields = {'process_type': model.TYPES.root}


@register('process')
class SubProcessLookup(NameLookup):
    model = models.Process
    base_lookup_fields = {'process_type': model.TYPES.subprocess}
    lookup_fields = NameLookup.lookup_fields + ('parent__name__icontains',)
    select_related = ('parent',)


@register('account')
class AccountsLookup(NameLookup):
    model = models.Account


@register('legal')
class LegalLookup(NameLookup):
    model = models.Legal


@register('assertion')
class AssertionLookup(NameLookup):
    model = models.Assertion


@register('system')
class SystemLookup(NameLookup):
    model = models.System


@register('service')
class ServiceLookup(NameLookup):
    model = models.Service


@register('risk')
class RiskLookup(NameLookup):
    model = models.Risk
    lookup_fields = NameLookup.lookup_fields + ('number__icontains', )


@register('control')
class ControlLookup(NameLookup):
    model = models.Control
    lookup_fields = NameLookup.lookup_fields + ('number__icontains', )


@register('control_plan')
class ControlPlanLookup(BaseLookup):
    model = models.ControlPlan
    select_related = 'control',
    prefetch_related = 'business_unit', 'system', 'service',


@register('control_step')
class ControlStepLookup(BaseLookup):
    model = models.ControlStep
    lookup_fields = BaseLookup.lookup_fields + ('step__icontains', )


@register('ipe')
class IPELookup(NameLookup):
    model = models.IPE


@register('business_unit')
class BusinessUnitLookup(NameLookup):
    model = models.BusinessUnit


@register('control_test')
class ControlTestLookup(BaseLookup):
    model = models.ControlTest


@register('deficiency')
class DeficiencyLookup(BaseLookup):
    model = models.Deficiency
    prefetch_related = ('control_test__control_plan', 'control_test__control_plan__control')
    lookup_fields = BaseLookup.lookup_fields + (
        'short_description__icontains',
        'control_test__control_plan__control__number__icontains',
        'control_test__control_plan__control__name__icontains',
    )


@register('abc_service')
class ABCServiceLookup(LookupChannel):
    min_length = 3

    def get_query(self, q, request):
        return get_abc_services(q)
