from django.conf import settings
from django.db import models
from django.utils.functional import cached_property
from django.utils.translation import ugettext_noop as _
from model_utils import Choices

from intranet.audit.src.core.models_bases import (
    CoreObject,
    UniqueNamedObject,
    NamedObject,
    BaseObject,
)
from intranet.audit.src.core.models_mixins import StatusMixin, ReviewerMixin


class System(CoreObject):
    pass


class Service(CoreObject):
    pass


class Process(UniqueNamedObject):

    TYPES = Choices(
        # Translators: Выводится в админке сущности Process, как выбор в поле
        # process_type для значения root
        ('root', _('корневой процесс'),
         ),
        # Translators: Выводится в админке сущности Process, как выбор в поле process_type
        # для значения subprocess
        ('subprocess', _('подпроцесс'),
         ),
    )
    process_type = models.CharField(max_length=50, choices=TYPES,)
    parent = models.ForeignKey('self', null=True, blank=True,
                               related_name='parent_process',
                               limit_choices_to={'process_type': TYPES.root},
                               )

    def __str__(self):
        # TODO Поменять этот метод когда перейдем на нормальный фронт, чтобы не было лишних запросов
        parent = '{} - '.format(self.parent.name) if self.parent else ''
        return '{}{}' .format(parent, self.name)


class Account(UniqueNamedObject):
    parent = models.ForeignKey('self', null=True, blank=True,
                               related_name='parent_account',
                               )


class Legal(UniqueNamedObject):
    pass


class BusinessUnit(UniqueNamedObject):
    pass


class Risk(NamedObject):
    number = models.CharField(
        max_length=100,
        unique=True,
        # Translators: Выводится в админке модели Risk в качестве пояснения к полю number
        help_text=_('Номер риска'),
    )

    def __str__(self):
        return '{} - {}'.format(self.number, self.name)


class Assertion(UniqueNamedObject):
    def __str__(self):
        return self.name


class Control(NamedObject):
    number = models.CharField(
        max_length=100,
        unique=True,
        # Translators: Выводится в админке модели Control в качестве пояснения к полю number
        help_text=_('Номер контрольной процедуры'),
    )

    def __str__(self):
        return '{} - {}'.format(self.number, self.name)


class IPE(StatusMixin, ReviewerMixin, NamedObject):
    APPLIANCES = Choices(
        # Translators: Выводится в админке модели IPE в качестве выбора значения у поля appliance
        ('control', _('control'),
         ),
        # Translators: Выводится в админке модели IPE в качестве выбора значения у поля appliance
        ('testing', _('testing'),
         ),
        # Translators: Выводится в админке модели IPE в качестве выбора значения у поля appliance
        ('control_testing', _('control/testing'),
         ),
    )
    TYPES = Choices(
        # Translators: Выводится в админке модели IPE в качестве выбора значения у поля ipe_type
        ('standart', _('Standart'),
         ),
        # Translators: Выводится в админке модели IPE в качестве выбора значения у поля ipe_type
        ('custom', _('Custom'),
         ),
    )
    service_description = models.CharField(null=True, blank=True,
                                           # Translators: Выводится в админке модели IPE в качестве
                                           # пояснения к полю service_description
                                           help_text=_('Сервис ABC'),
                                           max_length=500,
                                           )
    appliance = models.CharField(max_length=25, choices=APPLIANCES,
                                 null=True, blank=True,
                                 )
    source_data = models.TextField(null=True, blank=True,)
    report_logic = models.TextField(null=True, blank=True,)
    system = models.ForeignKey(System, blank=True, null=True, )
    comment = models.TextField(null=True, blank=True,)
    evidence = models.ManyToManyField('files.File', blank=True, verbose_name='Attach', )
    ipe_type = models.CharField(max_length=25, choices=TYPES,
                                null=True, blank=True,
                                verbose_name='Type',
                                )
    date_of_last_change = models.DateField(null=True, blank=True,
                                           # Translators: Выводится в админке в качестве названия
                                           # поля date_of_last_change для IPE
                                           verbose_name=_('Дата последнего изменения отчета'),
                                           )


class ControlPlan(ReviewerMixin, BaseObject):
    # Translators: Выводится в админке модели ControlPlan, если отсутствуют связанные сущности
    # business_units/system/service
    EMPTY_VALUE = _('none')
    METHODS = Choices(
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # method для manual
        ('manual', _('Ручной'),
         ),
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # method для auto
        ('auto', _('Автоматический'),
         ),
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # method для it_dependent
        ('it_dependent', _('Комбинированный'),
         ),
    )
    TYPES = Choices(
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # control_type для warning
        ('warning', _('Предупреждающая'),
         ),
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # control_type для revealing
        ('revealing', _('Выявляющая'),
         ),
    )
    FREQUENCIES = Choices(
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # frequency для yearly
        ('yearly', _('Eжегодно'),
         ),
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # frequency для half yearly
        ('half_yearly', _('Раз в полгода'),
         ),
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # frequency для quarterly
        ('quarterly', _('Ежеквартально'),
         ),
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # frequency для monthly
        ('monthly', _('Ежемесячно'),
         ),
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # frequency для weekly
        ('weekly', _('Еженедельно'),
         ),
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # frequency для daily
        ('daily', _('Ежедневно'),
         ),
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # frequency для many times per day
        ('many_times_per_day', _('Несколько раз в день'),
         ),
        # Translators: Выводится в админке модели ControlPlan в качестве выбора значения у поля
        # frequency для adhoc
        ('adhoc', _('Ad-Hoc'),
         ),
    )
    STATUSES = Choices(
        ('draft', _('Draft')),
        ('review', _('Review')),
        ('active', _('Active')),
        ('archived', _('Archived')),
        ('deleted', _('Deleted')),
    )

    status = models.CharField(max_length=25, choices=STATUSES, default=STATUSES.draft)

    process = models.ManyToManyField(Process)

    service = models.ManyToManyField(Service, blank=True,)
    system = models.ManyToManyField(System, blank=True,)

    risk = models.ManyToManyField(Risk)
    control = models.ForeignKey(Control)

    account = models.ManyToManyField(Account, blank=True,)
    legal = models.ManyToManyField(Legal, blank=True,)
    business_unit = models.ManyToManyField(BusinessUnit, blank=True,)

    reliance = models.BooleanField(default=False)
    key_control = models.BooleanField(default=True)
    to_test = models.BooleanField(default=True)
    comment = models.TextField(null=True, blank=True,)

    method = models.CharField(max_length=25,
                              choices=METHODS,
                              )
    frequency = models.CharField(max_length=25,
                                 # Translators: Выводится в админке ControlPlan как поясняющее
                                 # значения для поля frequency
                                 help_text=_('Частота выполнения контроля'),
                                 choices=FREQUENCIES,
                                 )
    control_type = models.CharField(max_length=25,
                                    choices=TYPES,
                                    # Translators: Выводится в админке ControlPlan как поясняющее
                                    # значения для поля control_type
                                    help_text=_('Тип контроля'),
                                    )

    antifraud = models.BooleanField(default=False,
                                    # Translators: Выводится в админке модели ControlPlan в качестве пояснения к
                                    #  полю antiafruad
                                    help_text=_('Противодействие мошенничеству'),
                                    )

    assertion = models.ManyToManyField(Assertion, blank=True,)
    evidence = models.TextField(
        # Translators: Выводится в админке ControlPlan как поясняющее значения для поля evidence
        help_text=_('Доказательство выполнения контроля'),
        null=True, blank=True,
    )
    regulation = models.TextField(
        # Translators: Выводится в админке ControlPlan как поясняющее значения для поля regulation
        help_text=_('Наименование внутреннего нормативного документа'),
        null=True, blank=True,
    )

    owner = models.ManyToManyField('users.StatedPerson', blank=True,
                                   related_name='control_owner',
                                   )

    description = models.TextField(
        # Translators: Выводится в админке ControlPlan как поясняющее значения для поля description
        help_text=_('Полное описание контроля в контексте плана'),
        null=True, blank=True,
        verbose_name='full description',
    )

    test_period_started = models.DateField(null=True, blank=True,
                                           # Translators: Выводится в админке в качестве названия
                                           # поля test_period_started для ControlPlan
                                           verbose_name=_('Plan period started'),
                                           )
    test_period_finished = models.DateField(null=True, blank=True,
                                            # Translators: Выводится в админке в качестве названия
                                            # поля test_period_finished для ControlPlan
                                            verbose_name=_('Plan period finished'),
                                            )

    def __str__(self):
        # TODO Поменять этот метод когда перейдем на нормальный фронт, чтобы не было лишних запросов
        return self.display_look

    @cached_property
    def has_related_tests(self):
        return self.controltest_set.exists()

    @cached_property
    def display_look(self):
        template = '{} - {} (BU: {},\n Service: {},\n System: {},\n Period: {} - {})\n'
        control = self.control

        business_units = self.business_unit.all()
        business_units = '[{}]'.format(','.join(business_unit.name for
                                                business_unit in business_units)) if business_units else self.EMPTY_VALUE
        systems = self.system.all()
        systems = '[{}]'.format(','.join(system.name for
                                         system in systems)) if systems else self.EMPTY_VALUE

        services = self.service.all()
        services = '[{}]'.format(','.join(service.name for
                                          service in services)) if services else self.EMPTY_VALUE

        date_from = self.test_period_started
        date_to = self.test_period_finished

        return template.format(control.number, control.name, business_units,
                               services, systems,
                               date_from.strftime(settings.DATE_DISPLAY_FORMAT)
                               if date_from else self.NO_DATE,
                               date_to.strftime(settings.DATE_DISPLAY_FORMAT)
                               if date_to else self.NO_DATE,
                               )


class ControlStep(BaseObject):
    RESULTS = Choices(
        # Translators: Выводится в админке ControlStep в качестве возможного
        # выбора для поля result (ok)
        ('ok', _('OK'),
         ),
        # Translators: Выводится в админке ControlStep в качестве возможного
        # выбора для поля result (not_ok)
        ('not_ok', _('NOT OK'),
         ),
    )
    step = models.TextField(
        # Translators: Выводится в админке ControlStep как поясняющее значения для поля step
        help_text=_('Описание шага')
    )
    comment = models.TextField(
        # Translators: Выводится в админке ControlStep как поясняющее значения для поля comment
        help_text=_('Комментарий'),
        blank=True,
        null=True,
    )
    result = models.CharField(max_length=10, choices=RESULTS, null=True,
                              blank=True,
                              # Translators: Выводится в админке ControlStep как поясняющее значения
                              # для поля result
                              help_text=_('Результат'),
                              )
    control_test = models.ForeignKey('core.ControlTest', null=True, blank=True)
    file = models.ManyToManyField('files.File', blank=True, )

    def __str__(self):
        return ''

    def to_excel(self):
        template = 'Step: {},\n Comment: {},\n Result: {}'
        return template.format(self.step, self.comment, self.result,)

    class Meta:
        order_with_respect_to = 'control_test'


class DeficiencyBase(BaseObject):
    """
    Базовая модель для недостатков и для их групп (агрегированных недостатков)
    """
    PROBABILITIES = Choices(
        # Translators: Выводится в админке Deficiency в качестве
        # возможного выбора для поля
        # misstatement_probability (low)
        ('low', _('low')),
        # Translators: Выводится в админке Deficiency в качестве
        # возможного выбора для поля
        # misstatement_probability (medium)
        ('medium', _('medium')),
        # Translators: Выводится в админке Deficiency в качестве
        # возможного выбора для поля
        # misstatement_probability (high)
        ('high', _('high')),
    )

    SIGNIFICANCE_EVALUATIONS = Choices(
        ('d', _('D')),
        ('sd', _('SD')),
        ('mw', _('MW')),
    )

    full_description = models.TextField(
        # Translators: Выводится в админке Deficiency как
        # поясняющее значения для поля full_description
        help_text=_('Подробное описание'),
        null=True,
        blank=True,
    )
    potential_impact = models.CharField(
        max_length=10,
        choices=PROBABILITIES,
        null=True,
        blank=True,
        # Translators: Выводится в админке Deficiency как поясняющее
        # значения для поля potential_impact
        help_text=_('Потенциальная величина искажения'),
    )
    misstatement_probability = models.CharField(
        max_length=10,
        choices=PROBABILITIES,
        null=True,
        blank=True,
        # Translators: Выводится в админке Deficiency как поясняющее значения
        # для поля misstatement_probability
        help_text=_('Вероятность искажения'),
    )
    mitigating_factors = models.TextField(
        null=True,
        blank=True,
        # Translators: Выводится в админке Deficiency как поясняющее значения
        # для поля mitigating_factors
        help_text=_('Факторы, снижающие риск искажения'),
    )
    significance_evaluation = models.CharField(
        max_length=10,
        choices=SIGNIFICANCE_EVALUATIONS,
        null=True,
        blank=True,
        # Translators: Выводится в админке Deficiency как поясняющее значения
        # для поля significance_evaluation
        help_text=_('Оценка значимости недостатка'),
    )
    comment = models.TextField(null=True, blank=True)
    estimate_closing_date = models.DateField(
        null=True,
        blank=True,
        # Translators: Выводится в админке / ответе апи
        # модели Deficiency как поясняющее значения
        # для поля estimate_closing_date
        help_text=_('Предполагаемая дата закрытия'),
    )

    class Meta:
        abstract = True


class Deficiency(DeficiencyBase):
    STATES = Choices(
        # Translators: Выводится в админке Deficiency в качестве
        # возможного выбора для поля state (open)
        ('open', _('open')),
        # Translators: Выводится в админке Deficiency в качестве
        # возможного выбора для поля state (fixed)
        ('fixed', _('fixed')),
        # Translators: Выводится в админке Deficiency в качестве
        # возможного выбора для поля state (deleted)
        ('deleted', _('deleted')),
    )

    INTERNAL_CONTROL_COMPONENTS = Choices(
        ('ce', _('Control Environment')),
        ('ca', _('Control Activities')),
        ('ic', _('Information and Communication')),
        ('m', _('Monitoring')),
        ('ra', _('Risk Assessment')),
    )

    short_description = models.TextField(
        # Translators: Выводится в админке Deficiency как поясняющее
        # значения для поля short_description
        help_text=_('Краткое описание'),
    )
    state = models.CharField(
        max_length=10,
        choices=STATES,
        default=STATES.open,
    )
    ticket_key = models.TextField(
        null=True,
        blank=True,
        verbose_name='startrek ticket',
    )
    root_cause_of_the_control_deficiency = models.TextField(null=True, blank=True)
    internal_control_component = models.CharField(
        max_length=50,
        choices=INTERNAL_CONTROL_COMPONENTS,
        null=True,
        blank=True,
    )
    need_to_be_aggregated = models.BooleanField(default=False)
    indication_of_other_deficiencies = models.TextField(null=True, blank=True)
    factors_in_evaluating_severity = models.TextField(null=True, blank=True)
    application_controls = models.TextField(null=True, blank=True)
    impact_of_gitc_deficiency = models.TextField(
        null=True,
        blank=True,
        help_text=_('Impact of GITC deficiency on each affected application control'),
    )

    def __str__(self):
        str_data = []
        for control_test in self.control_test.all():
            if control_test.control_plan and control_test.control_plan.control:
                str_data.append(str(control_test.control_plan.control))
            str_data.append(str(control_test))
        str_data.append(self.short_description)
        return ' / '.join(str_data)


class DeficiencyGroup(DeficiencyBase):
    """
    Группа недостатков. Она же – агрегированный недостаток
    """
    STATES = Choices(
        ('open', _('open')),
        ('fixed', _('fixed')),
        ('incorrect', _('incorrect')),
    )

    deficiencies = models.ManyToManyField(
        to=Deficiency,
        related_name='groups',
        blank=True,
    )
    state = models.CharField(
        max_length=16,
        choices=STATES,
        default=STATES.open,
    )

    def __str__(self):
        return '{} {}'.format(self.id, self.full_description)


class ControlTest(ReviewerMixin, BaseObject):

    STATUSES = Choices(
        ('draft', _('Draft')),
        ('review', _('Review')),
        # Note: для тестов неактуален статус "активный",
        # здесь больше подходит статус "tested".
        # Пока оставляем значение в БД "active",
        # чтобы не менять везде. Но когда-нибудь это нужно изменить.
        ('active', _('Tested')),
        ('archived', _('Archived')),
        ('deleted', _('Deleted')),
    )
    DEFAULT_STATUS = STATUSES.draft

    DESIGN_EFFICIENCY = Choices(
        # Translators: Выводится в админке ControlTest в качестве возможного
        # выбора для полей design_efficiency (effective)
        ('effective', _('Effective'),
         ),
        # Translators: Выводится в админке ControlTest в качестве возможного
        # выбора для полей design_efficiency (ineffective)
        ('ineffective', _('Ineffective'),
         ),
    )

    OPERATIONAL_EFFICIENCY = Choices(
        # Translators: Выводится в админке ControlTest в качестве возможного
        # выбора для полей operational_efficiency (effective)
        ('effective', _('Effective'),
         ),
        # Translators: Выводится в админке ControlTest в качестве возможного
        # выбора для полей operational_efficiency (ineffective)
        ('ineffective', _('Ineffective'),
         ),
        # Translators: Выводится в админке ControlTest в качестве возможного
        # выбора для полей operational_efficiency (not available)
        ('n_a', _('N/A'),
         ),
    )

    ADJUSTED_RISK = Choices(
        ('low', _('Low')),
        ('medium', _('Medium')),
        ('high', _('High')),
    )

    control_plan = models.ForeignKey(ControlPlan)
    status = models.CharField(max_length=25, choices=STATUSES, default=DEFAULT_STATUS)

    test_period_started = models.DateField(null=True, blank=True, )
    test_period_finished = models.DateField(null=True, blank=True, )
    testing_date = models.DateField(null=True, blank=True, )

    tester = models.ManyToManyField('users.StatedPerson', blank=True,
                                    related_name='tester',
                                    )
    sampling = models.TextField(null=True, blank=True,)

    evidence = models.ManyToManyField('files.File', blank=True)

    evidence_comments = models.TextField(null=True, blank=True)
    deficiency = models.ManyToManyField(Deficiency, blank=True, related_name='control_test')
    design_efficiency = models.CharField(max_length=50,
                                         choices=DESIGN_EFFICIENCY,
                                         null=True, blank=True,
                                         )
    operational_efficiency = models.CharField(max_length=50,
                                              choices=OPERATIONAL_EFFICIENCY,
                                              null=True, blank=True,
                                              )

    comment = models.TextField(null=True, blank=True,)
    review_date = models.DateField(null=True, blank=True)
    adjusted_risk = models.CharField(
        choices=ADJUSTED_RISK,
        max_length=50,
        null=True,
        blank=True,
    )
    mrc = models.BooleanField(
        default=False,
        help_text=_(
            'Use of judgement, including steps to identify and investigate significant '
            'differences from expectations and conclusions reached'
        ),
    )
    roll_forward = models.BooleanField(default=False)
    threshold_for_investigation = models.TextField(null=True, blank=True, )
    how_precision_is_affected = models.TextField(
        null=True,
        blank=True,
        help_text=_(
            'How does the level of predictability, level of aggregation, '
            'consistency of control performance and criteria for investigation affect precision'
        ),
    )
    how_management_identifies = models.TextField(
        null=True,
        blank=True,
        help_text=_(
            'Describe how management identifies, '
            'investigates and resolves deviations or differences'
        ),
    )

    def __str__(self):
        date_from = self.test_period_started
        date_to = self.test_period_finished
        return '{} - {}'.format(date_from.strftime(settings.DATE_DISPLAY_FORMAT)
                                if date_from else self.NO_DATE,
                                date_to.strftime(settings.DATE_DISPLAY_FORMAT)
                                if date_to else self.NO_DATE,
                                )


class ControlTestIPE(BaseObject):
    STATUS = Choices(
        # Translators: Выводится в админке модели ControlTestIPE в качестве выбора значения у поля status
        ('ok', _('OK'),
         ),
        # Translators: Выводится в админке модели ControlTestIPE в качестве выбора значения у поля status
        ('not_ok', _('Not OK'),
         ),
    )
    ipe = models.ForeignKey(IPE, related_name='control_test_ipe')
    control_test = models.ForeignKey(ControlTest, related_name='control_test_ipe')

    status = models.CharField(max_length=25, choices=STATUS,
                              null=True, blank=True,
                              )

    report_parameters = models.TextField(null=True, blank=True, )
    attach = models.ManyToManyField('files.File', blank=True, )

    class Meta:
        order_with_respect_to = 'control_test'
