from django.db import models
from django.utils.functional import cached_property
from django.conf import settings
from django.utils.translation import ugettext

from model_utils.models import TimeStampedModel, SoftDeletableModel


class BaseObject(SoftDeletableModel, TimeStampedModel):
    # Translators: Выводится в админке, если у сущности не заданы поля типа date
    NO_DATE = ugettext('Not specified')
    author = models.ForeignKey(settings.AUTH_USER_MODEL, null=False, blank=False,)

    def __str__(self):
        alias = self.alias
        try:
            alias += ' ' + self._repr_field
        except NotImplementedError:
            pass
        return alias

    @property
    def _repr_field(self):
        raise NotImplementedError

    @cached_property
    def alias(self):
        prefix = '{} {}'.format(self.class_name, self.id)
        return '{prefix} {created}'.format(created=self.creation_date,
                                           prefix=prefix,
                                           )

    @cached_property
    def filename(self):
        return '_'.join(self.alias.split())

    @cached_property
    def creation_date(self):
        return self.created.date().isoformat()

    @cached_property
    def class_name(self):
        return self.__class__.__name__

    def format_for_display(self, attr_name):
        value = getattr(self, attr_name)
        if len(value) < 20:
            return value
        return '{}...'.format(value[:20])

    class Meta:
        abstract = True


class NamedObject(BaseObject):
    name = models.CharField(max_length=1000,)

    def __str__(self):
        return self.name

    class Meta:
        abstract = True


class UniqueNamedObject(BaseObject):
    name = models.CharField(max_length=1000,
                            unique=True,
                            )

    def __str__(self):
        return self.name

    @property
    def api_related_display(self):
        return '{}: {}'.format(self.id, self.name)

    class Meta:
        abstract = True


class CoreObject(UniqueNamedObject):

    service_description = models.CharField(null=True, blank=True,
                                           # Translators: Выводится в админке в качестве пояснения
                                           # к полю service_description
                                           help_text=ugettext('Сервис ABC'),
                                           max_length=500,
                                           )

    class Meta:
        abstract = True
