import logging

from django.core.exceptions import ObjectDoesNotExist
from django.db.models.fields import NOT_PROVIDED
from django.db.models.manager import Manager
from django.utils.translation import ugettext
from import_export import fields, resources, widgets

from intranet.audit.src.core import models
from intranet.audit.src.users.models import User


logger = logging.getLogger(__name__)


def get_field_help_text(model, field_name):
    return next(f.help_text for f in model._meta.fields if f.name == field_name)


class ChoicesWidget(widgets.Widget):
    def __init__(self, choices, *args, **kwargs):
        self.choices = dict(choices)
        self.revert_choices = dict((v, k) for k, v in self.choices.items())

    def clean(self, value, row=None, *args, **kwargs):
        return self.revert_choices.get(value, value) if value else None

    def render(self, value, obj=None):
        return self.choices.get(value, '')


class BooleanWidget(widgets.BooleanWidget):
    """
    Widget for converting boolean fields.
    """
    # Translators: True/False для вывода в файл при экспорте
    TRUE_VALUES = [ugettext('True'), 1]
    FALSE_VALUE = ugettext('False')

widgets.BooleanWidget = BooleanWidget


class DateWidget(widgets.DateWidget):
    def render(self, value, obj=None):
        if not value:
            return ''
        return value.strftime('%d.%m.%Y')

widgets.DateWidget = DateWidget


class ForeignKeyWidget(widgets.ForeignKeyWidget):
    def render(self, value, obj=None):
        if value is None:
            return ""

        return getattr(value, self.field) if self.field != 'pk' else str(value)


class StatedPersonWidget(widgets.Widget):
    def render(self, value, obj=None):
        if value is None:
            return ""
        if not isinstance(value, str):
            value = str(value)
        return '\015'.join(person.split('-')[0] for person in value.split('\015'))


class BaseResource(resources.ModelResource):
    def export(self, queryset=None, *args, **kwargs):
        query = getattr(queryset, 'query', None)
        self.lookup_id = None
        if query:
            for children in query.where.children:
                if (children.lhs.alias == 'core_controlplan' and
                            children.lhs.target.get_attname() == 'process_id'):
                    self.lookup_id = children.rhs
                    break

        return super().export(queryset, *args, **kwargs)

    def get_fields(self, **kwargs):
        for name, field in self.fields.items():
            field.lookup_id = self.lookup_id
            if field.column_name != field.column_name.upper():
                field.column_name = ' '.join(field.column_name.title().split('_'))
        return super().get_fields(**kwargs)


class ManyToManyField(fields.Field):
    def __init__(self, attribute=None, column_name=None, widget=None,
                 default=NOT_PROVIDED, readonly=False, field_name=None,
                 separator='\015', first_from_many=False):
        self.field_name = field_name
        self.separator = separator
        self.first_from_many = first_from_many
        super().__init__(attribute, column_name, widget,
                         default, readonly,
                         )

    def get_value(self, obj):
        attrs = self.attribute.split('__')
        value = obj

        for attr in attrs:
            try:
                value = getattr(value, attr, None)
                if isinstance(value, Manager):
                    if self.first_from_many:
                        value = value.first()
                    else:
                        value = self.separator.join(
                            getattr(obj, self.field_name)
                            if self.field_name else str(obj)
                            for obj in value.all()
                        )
            except (ValueError, ObjectDoesNotExist):
                return None
            if value is None:
                return None

        if callable(value) and not isinstance(value, Manager):
            value = value()
        return value


class ChainField(fields.Field):
    def __init__(self, attribute=None, column_name=None, widget=None,
                 default=NOT_PROVIDED, readonly=False, lookup_value=None,
                 query_field_name=None):
        self.lookup_value = lookup_value
        self.query_field_name = query_field_name
        super().__init__(attribute, column_name, widget,
                         default, readonly,
                         )

    def get_value(self, obj):
        attrs = self.attribute.split('__')
        value = obj

        for attr in attrs:
            try:
                value = getattr(value, attr, None)
                if (isinstance(value, Manager) and
                            value.query_field_name == self.query_field_name):
                    if self.lookup_id and self.lookup_value:
                        query = {self.lookup_value: self.lookup_id}
                        value = value.get(**query)

                    else:
                        logger.error('Didn\'t get needed parameters for "%s", "%s", got "%s", "%s"',
                                     value, attr, self.lookup_value, self.lookup_id)
                        value = value.first()

            except (ValueError, ObjectDoesNotExist):
                return None
            if value is None:
                return None

        if callable(value) and not isinstance(value, Manager):
            value = value()
        return value


class YesNoField(fields.Field):
    def get_value(self, obj):
        value = super().get_value(obj)
        if value is True:
            return 'Yes'
        elif value is False:
            return 'No'


class ControlPlanResource(BaseResource):
    # Translators: Название колонки control при экспорте ControlPlan
    control = fields.Field(
        column_name=ugettext('Контроль'),
        attribute='control',
        widget=ForeignKeyWidget(models.Control),
    )
    key_control = fields.Field(column_name=ugettext('Ключевой Контроль'),
                               attribute='key_control',
                               )
    method = fields.Field(column_name=ugettext('Метод Контроля'),
                          attribute='method',
                          widget=ChoicesWidget(models.ControlPlan.METHODS),
                          )
    frequency = fields.Field(column_name=ugettext('Частота Контроля'),
                             attribute='frequency',
                             widget=ChoicesWidget(models.ControlPlan.FREQUENCIES),
                             )
    control_type = fields.Field(column_name=ugettext('Тип Контроля'),
                                attribute='control_type',
                                widget=ChoicesWidget(models.ControlPlan.TYPES),
                                )
    # Translators: Название колонки risk при экспорте ControlPlan
    risk = ManyToManyField(column_name=ugettext('Risks'), attribute='risk')
    testing_dates = fields.Field(column_name=ugettext('Даты процедуры'))

    process = ManyToManyField(column_name=ugettext('Процесс'))

    # Translators: Название колонки legal при экспорте ControlPlan
    legal = ManyToManyField(column_name=ugettext('ЮЛ'), attribute='legal')
    # Translators: Название колонки business_unit при экспорте ControlPlan
    business_unit = ManyToManyField(column_name=ugettext('БЮ'), attribute='business_unit')
    # Translators: Название колонки system при экспорте ControlPlan
    system = ManyToManyField(column_name=ugettext('Система'), attribute='system')
    # Translators: Название колонки service при экспорте ControlPlan
    service = ManyToManyField(column_name=ugettext('Сервис'), attribute='service')
    # Translators: Название колонки account при экспорте ControlPlan
    account = ManyToManyField(column_name=ugettext('Account'), attribute='account')
    # Translators: Название колонки assertion при экспорте ControlPlan
    assertion = ManyToManyField(column_name=ugettext('Assertion'), attribute='assertion')
    # Translators: Название колонки owner при экспорте ControlPlan
    owner = ManyToManyField(column_name=ugettext('Control Procedure Owner'), attribute='owner',
                            widget=StatedPersonWidget())

    status = fields.Field(column_name=ugettext('Status'), attribute='status',
                          widget=ChoicesWidget(models.ControlPlan.STATUSES),
                          )

    reviewer = ManyToManyField(column_name=ugettext('Reviewer'), attribute='reviewer',
                               widget=StatedPersonWidget(),
                               )

    def dehydrate_testing_dates(self, obj):
        return '{}/{}'.format(obj.test_period_started,
                              obj.test_period_finished,
                              )

    def dehydrate_process(self, obj):
        return '\n'.join(
            '{} -> {}'.format(getattr(process.parent, 'name', 'No parent'), process.name)
            for process in obj.process.all()
        )

    class Meta:
        model = models.ControlPlan
        exclude = ('id', 'created', 'is_removed',
                   'author', 'modified', 'to_test',
                   'test_period_started', 'test_period_finished',
                   'reliance',
                   )
        export_order = (
            'control',
            'key_control',
            'method',
            'frequency',
            'status',
            'control_type',
            'risk',
            'testing_dates',
            'process',
            'business_unit',
            'legal',
            'system',
            'service',
            'account',
            'assertion',
            'owner',
            'reviewer',
            'description',
            'evidence',
            'regulation',
            'antifraud',
            'comment',
        )


class ControlTestResource(BaseResource):
    control = fields.Field(
        column_name=ugettext('Контроль'),
        attribute='control_plan__control',
        widget=ForeignKeyWidget(models.Control),
    )
    key_control = fields.Field(column_name=ugettext('Ключевой Контроль'),
                               attribute='control_plan__key_control',
                               )
    method = fields.Field(column_name=ugettext('Метод Контроля'),
                          attribute='control_plan__method',
                          widget=ChoicesWidget(models.ControlPlan.METHODS),
                          )
    frequency = fields.Field(column_name=ugettext('Частота Контроля'),
                             attribute='control_plan__frequency',
                             widget=ChoicesWidget(models.ControlPlan.FREQUENCIES),
                             )
    control_type = fields.Field(column_name=ugettext('Тип Контроля'),
                                attribute='control_plan__control_type',
                                widget=ChoicesWidget(models.ControlPlan.TYPES),
                                )
    risk = ManyToManyField(column_name=ugettext('Риск'), attribute='control_plan__risk',
                           widget=ForeignKeyWidget(models.Risk),
                           )

    testing_dates = fields.Field(column_name=ugettext('Даты проведения тестирования'))
    process = ManyToManyField(column_name=ugettext('Процесс'))

    business_unit = ManyToManyField(column_name=ugettext('БЮ'),
                                    attribute='control_plan__business_unit',
                                    widget=ForeignKeyWidget(models.BusinessUnit),
                                    )
    legal = ManyToManyField(column_name=ugettext('ЮЛ'),
                            attribute='control_plan__legal',
                            widget=ForeignKeyWidget(models.Legal),
                            )
    system = ManyToManyField(column_name=ugettext('Система'),
                             attribute='control_plan__system',
                             widget=ForeignKeyWidget(models.System),
                             )
    service = ManyToManyField(column_name=ugettext('Сервис'),
                              attribute='control_plan__service',
                              widget=ForeignKeyWidget(models.Service),
                              )
    account = ManyToManyField(column_name=ugettext('Account'),
                              attribute='control_plan__account',
                              widget=ForeignKeyWidget(models.Account),
                              )
    assertion = ManyToManyField(column_name=ugettext('Assertion'),
                                attribute='control_plan__assertion',
                                widget=ForeignKeyWidget(models.Assertion),
                                )
    owner = ManyToManyField(column_name=ugettext('Control Procedure Owner'),
                            attribute='control_plan__owner',
                            widget=StatedPersonWidget(),
                            )

    tester = ManyToManyField(column_name=ugettext('Tester'), attribute='tester',
                             widget=StatedPersonWidget(),
                             )

    evidence = fields.Field(column_name=ugettext('Evidence'), attribute='evidence_comments')
    # Translators: Название колонки control_steps при экспорте ControlTest
    control_steps = fields.Field(column_name=ugettext('Steps'))
    control_steps_statuses = fields.Field(column_name=ugettext('Steps Statuses'))
    control_steps_comments = fields.Field(column_name=ugettext('Steps Comments'))

    status = fields.Field(column_name=ugettext('Status'), attribute='status',
                          widget=ChoicesWidget(models.ControlTest.STATUSES),
                          )

    reviewer = ManyToManyField(column_name=ugettext('Reviewer'), attribute='reviewer',
                               widget=StatedPersonWidget(),
                               )

    operational_efficiency = fields.Field(column_name=ugettext('Operational Efficiency'),
                                          attribute='operational_efficiency',
                                          widget=ChoicesWidget(models.ControlTest.OPERATIONAL_EFFICIENCY),
                                          )
    testing_date = fields.Field(
        column_name=ugettext('Testing date'),
        attribute='testing_date',
    )
    review_date = fields.Field(
        column_name=ugettext('Review date'),
        attribute='review_date',
    )
    adjusted_risk = fields.Field(
        column_name=ugettext('Adjusted risk'),
        attribute='adjusted_risk',
        widget=ChoicesWidget(models.ControlTest.ADJUSTED_RISK),
    )
    mrc = YesNoField(column_name=ugettext('MRC'), attribute='mrc')
    roll_forward = YesNoField(column_name=ugettext('Roll-forward'), attribute='roll_forward')
    threshold_for_investigation = fields.Field(
        column_name=ugettext('Threshold for investigation'),
        attribute='threshold_for_investigation',
    )
    how_precision_is_affected = fields.Field(
        column_name=ugettext(get_field_help_text(models.ControlTest, 'how_precision_is_affected')),
        attribute='how_precision_is_affected',
    )
    how_management_identifies = fields.Field(
        column_name=ugettext(get_field_help_text(models.ControlTest, 'how_management_identifies')),
        attribute='how_management_identifies',
    )

    def dehydrate_process(self, obj):
        return '\n'.join(
            '{} -> {}'.format(getattr(process.parent, 'name', 'No parent'), process.name)
            for process in obj.control_plan.process.all()
        )

    def dehydrate_testing_dates(self, obj):
        return '{}/{}'.format(obj.test_period_started,
                              obj.test_period_finished,
                              )

    def _get_steps_data(self, obj, attr):
        return '\n'.join('{} - {}'.format(index, getattr(step, attr))
                         for index, step in enumerate(obj.controlstep_set.all(), 1)
                         )

    def dehydrate_control_steps(self, obj):
        return self._get_steps_data(obj, 'step')

    def dehydrate_control_steps_statuses(self, obj):
        return self._get_steps_data(obj, 'result')

    def dehydrate_control_steps_comments(self, obj):
        return self._get_steps_data(obj, 'comment')

    class Meta:
        model = models.ControlTest
        exclude = (
            'id',
            'created',
            'is_removed',
            'author',
            'modified',
            'file',
            'deficiency',
            'test_period_started',
            'test_period_finished',
            'control_plan',
            'evidence_comments',
        )
        export_order = (
            'control',
            'key_control',
            'method',
            'frequency',
            'control_type',
            'status',
            'reviewer',
            'review_date',
            'risk',
            'testing_dates',
            'process',
            'business_unit',
            'legal',
            'system',
            'service',
            'account',
            'assertion',
            'owner',
            'tester',
            'testing_date',
            'design_efficiency',
            'operational_efficiency',
            'sampling',
            'evidence',
            'comment',
            'control_steps',
            'control_steps_statuses',
            'control_steps_comments',
            'adjusted_risk',
            'mrc',
            'roll_forward',
            'threshold_for_investigation',
            'how_precision_is_affected',
            'how_management_identifies',
        )


class IPEResource(BaseResource):
    # Translators: Название колонки control при экспорте ControlPlan
    system = fields.Field(column_name=ugettext('Система'), attribute='system',
                          widget=ForeignKeyWidget(models.System))

    name = fields.Field(column_name=ugettext('IPE'), attribute='name')
    ipe_type = fields.Field(column_name=ugettext('Type'), attribute='ipe_type',
                            widget=ChoicesWidget(models.IPE.TYPES),
                            )
    appliance = fields.Field(column_name=ugettext('Appliance'), attribute='appliance',
                             widget=ChoicesWidget(models.IPE.APPLIANCES),
                             )
    author = fields.Field(column_name=ugettext('Author'), attribute='author',
                          widget=ForeignKeyWidget(User))

    status = fields.Field(column_name=ugettext('Status'), attribute='status',
                          widget=ChoicesWidget(models.IPE.STATUSES),
                          )

    reviewer = ManyToManyField(column_name=ugettext('Reviewer'), attribute='reviewer',
                               widget=StatedPersonWidget(),
                               )

    class Meta:
        model = models.IPE
        exclude = 'id', 'is_removed', 'modified', 'evidence', 'service_description',
        export_order = (
            'name', 'system', 'status', 'ipe_type', 'appliance', 'created',
            'author', 'reviewer', 'source_data', 'report_logic', 'comment'
        )


class DeficiencyResource(BaseResource):

    control = ManyToManyField(
        column_name=ugettext('Контроль'),
        attribute='control_test__control_plan__control',
        widget=ForeignKeyWidget(models.Control),
        first_from_many=True,
    )
    key_control = ManyToManyField(
        column_name=ugettext('Ключевой Контроль'),
        attribute='control_test__control_plan__key_control',
        first_from_many=True,
    )
    method = ManyToManyField(
        column_name=ugettext('Метод Контроля'),
        attribute='control_test__control_plan__method',
        first_from_many=True,
        widget=ChoicesWidget(models.ControlPlan.METHODS),
    )
    frequency = ManyToManyField(
        column_name=ugettext('Частота Контроля'),
        attribute='control_test__control_plan__frequency',
        first_from_many=True,
        widget=ChoicesWidget(models.ControlPlan.FREQUENCIES),
    )
    control_type = ManyToManyField(
        column_name=ugettext('Тип Контроля'),
        attribute='control_test__control_plan__control_type',
        first_from_many=True,
        widget=ChoicesWidget(models.ControlPlan.TYPES),
    )
    risk = ManyToManyField(
        column_name=ugettext('Риск'),
        attribute='control_test__control_plan__risk',
        widget=ForeignKeyWidget(models.Risk),
        first_from_many=True,
    )
    testing_dates = fields.Field(column_name=ugettext('Даты проведения тестирования'))
    process = ManyToManyField(column_name=ugettext('Процесс'))
    business_unit = ManyToManyField(
        column_name=ugettext('БЮ'),
        attribute='control_test__control_plan__business_unit',
        widget=ForeignKeyWidget(models.BusinessUnit),
        first_from_many=True,
    )
    legal = ManyToManyField(
        column_name=ugettext('ЮЛ'),
        attribute='control_test__control_plan__legal',
        widget=ForeignKeyWidget(models.Legal),
        first_from_many=True,
    )
    system = ManyToManyField(
        column_name=ugettext('Система'),
        attribute='control_test__control_plan__system',
        widget=ForeignKeyWidget(models.System),
        first_from_many=True,
    )
    service = ManyToManyField(
        column_name=ugettext('Сервис'),
        attribute='control_test__control_plan__service',
        widget=ForeignKeyWidget(models.Service),
        first_from_many=True,
    )
    account = ManyToManyField(
        column_name=ugettext('Account'),
        attribute='control_test__control_plan__account',
        widget=ForeignKeyWidget(models.Account),
        first_from_many=True,
    )
    assertion = ManyToManyField(
        column_name=ugettext('Assertion'),
        attribute='control_test__control_plan__assertion',
        widget=ForeignKeyWidget(models.Assertion),
        first_from_many=True,
    )
    owner = ManyToManyField(
        column_name=ugettext('Control Procedure Owner'),
        attribute='control_test__control_plan__owner',
        widget=StatedPersonWidget(),
        first_from_many=True,
    )
    tester = ManyToManyField(
        column_name=ugettext('Tester'),
        attribute='control_test__tester',
        widget=StatedPersonWidget(),
        first_from_many=True,
    )
    design_efficiency = ManyToManyField(
        column_name=ugettext('Design Efficiency'),
        attribute='control_test__design_efficiency',
        first_from_many=True,
    )
    operational_efficiency = ManyToManyField(
        column_name=ugettext('Operational Efficiency'),
        attribute='control_test__operational_efficiency',
        first_from_many=True,
        widget=ChoicesWidget(models.ControlTest.OPERATIONAL_EFFICIENCY),
    )
    author = fields.Field(
        column_name=ugettext('Author'),
        attribute='author',
        widget=ForeignKeyWidget(User),
    )
    ticket = fields.Field(column_name=ugettext('Ticket'), attribute='ticket_key')

    root_cause_of_the_control_deficiency = fields.Field(
        column_name=ugettext('Root cause of the control deficiency'),
        attribute='root_cause_of_the_control_deficiency',
    )
    internal_control_component = fields.Field(
        column_name=ugettext('Internal Control Component'),
        attribute='internal_control_component',
        widget=ChoicesWidget(models.Deficiency.INTERNAL_CONTROL_COMPONENTS),
    )
    need_to_be_aggregated = YesNoField(
        column_name=ugettext('Need to be aggregated'),
        attribute='need_to_be_aggregated',
    )
    indication_of_other_deficiencies = fields.Field(
        column_name=ugettext('Indication of other deficiencies'),
        attribute='indication_of_other_deficiencies',
    )
    factors_in_evaluating_severity = fields.Field(
        column_name=ugettext('Factors in evaluating severity'),
        attribute='factors_in_evaluating_severity',
    )
    application_controls = fields.Field(
        column_name=ugettext('Application controls'),
        attribute='application_controls',
    )
    impact_of_gitc_deficiency = fields.Field(
        column_name=ugettext(get_field_help_text(models.Deficiency, 'impact_of_gitc_deficiency')),
        attribute='impact_of_gitc_deficiency',
    )

    def dehydrate_process(self, obj):
        control_test = obj.control_test.first()
        if control_test:
            return '\n'.join(
                '{} -> {}'.format(process.parent.name, process.name)
                for process in control_test.control_plan.process.all()
            )

    def dehydrate_testing_dates(self, obj):
        control_test = obj.control_test.first()
        started = None
        finished = None
        if control_test:
            started = control_test.test_period_started
            finished = control_test.test_period_finished
        return '{}/{}'.format(started, finished)

    class Meta:
        model = models.Deficiency
        exclude = (
            'id',
            'is_removed',
            'modified',
            'ticket_key',
        )
        export_order = (
            'control',
            'key_control',
            'method',
            'frequency',
            'control_type',
            'risk',
            'testing_dates',
            'process',
            'business_unit',
            'legal',
            'system',
            'service',
            'account',
            'assertion',
            'owner',
            'tester',
            'design_efficiency',
            'operational_efficiency',
            'state',
            'created',
            'author',
            'potential_impact',
            'misstatement_probability',
            'significance_evaluation',
            'short_description',
            'full_description',
            'ticket',
            'mitigating_factors',
            'comment',
            'root_cause_of_the_control_deficiency',
            'internal_control_component',
            'need_to_be_aggregated',
            'indication_of_other_deficiencies',
            'factors_in_evaluating_severity',
            'estimate_closing_date',
            'application_controls',
            'impact_of_gitc_deficiency',
        )


class DeficiencyGroupResource(BaseResource):

    author = fields.Field(
        column_name=ugettext('Author'),
        attribute='author',
        widget=ForeignKeyWidget(User),
    )

    class Meta:
        model = models.DeficiencyGroup
        exclude = (
            'id',
            'is_removed',
            'modified',
        )
        export_order = (
            'created',
            'full_description',
            'mitigating_factors',
            'state',
            'significance_evaluation',
            'potential_impact',
            'misstatement_probability',
            'author',
            'estimate_closing_date',
            'comment',
        )
