from django.core import urlresolvers
from django.utils.functional import cached_property
from django.db import models
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext
from django.utils import timezone
from django.conf import settings

from intranet.audit.src.core.models import NamedObject

ALLOW_EXCEL = {'image/png', 'image/jpeg', 'image/x-ms-bmp', }

NAME_EXTEND_DATE_FORMAT = '%Y_%m_%d_%H_%M_%S'


class File(NamedObject):
    file = models.FileField(upload_to='audit', unique=True, max_length=1000, )
    size = models.IntegerField()
    content_type = models.CharField(max_length=300)

    @cached_property
    def file_link(self):
        if self.id:
            return mark_safe('<a href={}>{}</a>'.format(self.file_link_url,
                                                        self.name,
                                                        )
                             )
        # Translators: Выводится в админке если файл еще не был приложен к объекту File
        return ugettext('Файл пока не прикреплен')

    @cached_property
    def file_link_url(self):
        if self.id:
            return urlresolvers.reverse("files:file", kwargs={'pk': self.id})

    def save(self, *args, **kwargs):
        file_obj = self.file.file
        self.size = file_obj.size
        self.content_type = file_obj.content_type
        self.name = self.file.name
        self.file.name = self.create_unique_name(self.file.name)
        return super().save(*args, **kwargs)

    def create_unique_name(self, name):
        try:
            base, extension = name.rsplit('.', 1)
            extension = '.{}'.format(extension)
        except ValueError:
            base = name
            extension = ''
        unique_seed = timezone.now().strftime(NAME_EXTEND_DATE_FORMAT)
        return '{}_{}{}'.format(base, unique_seed, extension)

    def __str__(self):
        return '{} - {}'.format(self.name,
                                self.modified.strftime(settings.DATE_DISPLAY_FORMAT),
        )
