import uuid

from django.contrib import admin, messages
from django.contrib.admin.helpers import ActionForm
from django import forms

from . import models
from .dao.user import get_users_by_uids_from_staff


class CreateUser(ActionForm):
    user = forms.fields.CharField()


class UserAdmin(admin.ModelAdmin):
    search_fields = 'login',
    list_display = 'login', 'last_name', 'first_name', 'is_superuser'
    exclude = 'uid', 'password',
    readonly_fields = 'login', 'last_name', 'first_name', 'last_login',
    filter_horizontal = 'groups', 'user_permissions',

    def has_add_permission(self, request):
        return False

    action_form = CreateUser

    actions = ['create_user']

    def create_user(self, request, queryset):
        user_to_create = request.POST['user']
        users = get_users_by_uids_from_staff(user_to_create)
        user = users[user_to_create]
        messages.success(request, 'Create user {}'.format(user))


class StatedPersonAdmin(admin.ModelAdmin):
    search_fields = 'login', 'department', 'position',
    list_display = 'id', 'uid', 'login', 'department', 'last_name', 'first_name', 'position'
    exclude = 'last_name', 'first_name', 'department_slug', 'login', 'is_removed',
    readonly_fields = 'created', 'modified', 'position', 'id', 'uid',

    def has_change_permission(self, request, obj=None):
        if obj is not None and obj.position != models.StatedPerson.GROUP_POSITION_SLUG:
            return False
        return super().has_change_permission(request, obj=obj)

    def save_model(self, request, obj, form, change):
        current_id = getattr(obj, 'id', None)
        if not current_id:
            obj_id = '{}_{}'.format(uuid.uuid4().hex,
                                    models.StatedPerson.GROUP_POSITION_SLUG,
                                    )
            obj.position = models.StatedPerson.GROUP_POSITION_SLUG
            obj.id = obj_id
        obj.save()


admin.site.register(models.User, UserAdmin)
admin.site.register(models.StatedPerson, StatedPersonAdmin)

