from django.core.exceptions import ObjectDoesNotExist

from ..models import User, StatedPerson
from ..logic.user import (_get_users_objects, get_users_from_blackbox,
                          get_users_from_staff, get_stated_persons_from_staff,
                          _prepare_stated_persons, _get_check_key,
                          )

from django_yauth.util import get_real_ip

PERSON_ATTRS_TO_CHECK = ('uid', 'position', 'department_slug',
                         'department', 'last_name',
                         )


def get_user_by_request(request):
    try:
        user = User.objects.get(uid=request.yauser.uid)
    except ObjectDoesNotExist:
        user = _get_users_objects(request.yauser, model=User)[0]
    return user


def check_users_existence(*uids):
    users = {user.uid: user for user in
             User.objects.filter(uid__in=uids)
             }
    not_existing = tuple(uid for uid in uids if uid not in users)
    return users, not_existing


def check_stated_persons_existence(stated_persons):
    current_persons = {
        _get_check_key(person, PERSON_ATTRS_TO_CHECK): person
        for person in StatedPerson.objects.filter(uid__in=[
            stated_person.uid for stated_person in stated_persons
            ]
        )
    }
    already_exists = {}
    for stated_person in stated_persons:
        stated_person_key = _get_check_key(stated_person, PERSON_ATTRS_TO_CHECK)
        full_overlap_person = current_persons.get(stated_person_key)
        if full_overlap_person:
            already_exists[stated_person.uid] = full_overlap_person

    not_existing = tuple(person for person in stated_persons if
                         person.uid not in already_exists)
    return already_exists, not_existing


def create_stated_persons(stated_persons):
    _prepare_stated_persons(*stated_persons)
    return {
        stated_person.uid: stated_person for stated_person in
        StatedPerson.objects.bulk_create(stated_persons)
    }


def create_new_users(new_users):
    return {
        user.uid: user for user in
        User.objects.bulk_create(new_users)
    }


def get_stated_persons_by_uids_from_staff(*uids, request=None):
    query = 'uid in [{}]'.format(','.join(('"{}"'.format(uid) for uid in uids)))
    persons_from_staff = get_stated_persons_from_staff(query=query, request=request,)
    stated_persons, to_create = check_stated_persons_existence(persons_from_staff)
    stated_persons.update(create_stated_persons(to_create))
    return stated_persons


def get_users_by_uids_from_staff(*uids, request=None):
    users, to_create = check_users_existence(*uids)
    if to_create:
        query = 'uid in [{}]'.format(','.join(('"{}"'.format(uid) for uid in to_create)))
        new_users = get_users_from_staff(query=query, request=request,)
        users.update(create_new_users(new_users))

    return users


def get_users_by_uids_from_blackbox(*uids, request):
    users, to_create = check_users_existence(*uids)
    if to_create:
        user_ip = get_real_ip(request)
        new_users = get_users_from_blackbox(*to_create, user_ip=user_ip)
        users.update(create_new_users(new_users))

    return users


def get_group_stated_data(query):
    return StatedPerson.objects.filter(
        position=StatedPerson.GROUP_POSITION_SLUG,
        department__icontains=query,
    )
