import logging
import uuid

from django.conf import settings
from blackbox import JsonBlackbox

from ..models import User, StatedPerson
from .staff_person import get_staff_persons

logger = logging.getLogger(__name__)
BLACK_BOX_CALLER = JsonBlackbox(url=settings.BLACKBOX_URL)
DB_FIELDS = 'userinfo.firstname.uid', 'userinfo.lastname.uid',

VALUE_MAP = {
    'uid': ('uid', ),
    'login': ('login', ),
    'last_name': ('last_name', ),
    'first_name': ('first_name', ),
}

ATTRIBUTES = 'uid', 'login', 'last_name', 'first_name',


def get_user_info(*uids, user_ip):
    return BLACK_BOX_CALLER.userinfo(userip=user_ip,
                                     uid=uids,
                                     dbfields=DB_FIELDS,
                                     )


def prepare_blackbox_response(blackbox_response):
    prepared_response = list()

    users_info = blackbox_response.get('users', [])

    if not users_info:
        logger.warning('Got no users from blackbox "%s"', users_info)

    for user_info in users_info:
        user_uid = user_info['id']
        error = user_info.get('error')
        if error:
            logger.error('Got error "%s" while getting data with "%s"',
                         error, user_uid)
            continue

        user_login = user_info.get('login')
        if not user_login:
            logger.info('User with uid "%s" doesnt exists', user_uid)
            continue

        fields_info = user_info['dbfields']
        first_name = fields_info['userinfo.firstname.uid']
        last_name = fields_info['userinfo.lastname.uid']
        prepared_response.append({
            'uid': user_uid,
            'login': user_login,
            'first_name': first_name,
            'last_name': last_name,
        })
    return prepared_response


def get_users_from_blackbox(*uids, user_ip, **kwargs):
    blackbox_response = get_user_info(*uids, user_ip=user_ip,)

    users_data = prepare_blackbox_response(blackbox_response)

    return _get_users_objects(*users_data, model=User, **kwargs)


def get_stated_persons_from_staff(query, request=None):
    """
    Возращает объекты пользователей соответствующие
    переданному запросу

    :param query: str
    :param request: WSGIRequest
    :return: list
    """
    persons = get_staff_persons(query=query)
    return _get_users_objects(
        *persons,
        attributes=settings.STAFF_STATED_PERSON_ATTRIBUTES,
        value_map=settings.STAFF_STATED_PERSON_VALUE_MAP,
        model=StatedPerson,
    )


def get_users_from_staff(query, request=None,):
    """
    Возращает объекты пользователей соответствующие
    переданному запросу

    :param query: str
    :param request: WSGIRequest
    :return: list
    """
    persons = get_staff_persons(query=query)

    users_objects = _get_users_objects(*persons, value_map=settings.STAFF_VALUE_MAP, model=User)
    _prepare_user_objects(users_objects=users_objects)
    return users_objects


def _prepare_user_objects(users_objects):
    for user in users_objects:
        user.set_unusable_password()


def _prepare_stated_persons(*stated_persons):
    for stated_person in stated_persons:
        stated_person.id = '{}_{}'.format(
            stated_person.id,
            uuid.uuid4().hex,
        )


def _get_users_objects(*user_data, value_map=None, attributes=None,
                       model, **kwargs):
    """
    Получает на вход список объектов содержащих данные
    пользователей и преобразует их в объекты пользователей
    используя параметр value_map в качестве пути к нужным 
    данным
    
    """
    value_map = value_map or VALUE_MAP
    attributes = attributes or ATTRIBUTES

    users = []
    for user in user_data:
        params = {}
        for key in attributes:
            params[key] = _get_user_param(user, value_map[key])

        params.update(kwargs)
        user = model(**params)
        users.append(user)

    return users


def _get_user_param(user_data, path, index=0):
    """
    Получаем значение параметра исходя из
    переданного пути к нему. Если path = ('data', 'uid', 'value')
    То uid пользователя будет извлечен из user['data']['uid']['value']
    в случае если объект user является словарем, и из user.data.uid.value в
    прочих случаях
    >>> user_data = {'data': {'uid': {'value': {'id': 1234}}}}
    >>> _get_user_param(user_data, ('data', 'uid', 'value', 'id'))
    1234
    
    :param user_data: dict/yauser
    :param path: list/tuple
    :param index: int
    """
    if isinstance(user_data, dict):
        value = user_data[path[index]]
    else:
        value = getattr(user_data, path[index])
    index += 1
    if len(path) == index:
        return value
    return _get_user_param(value, path, index)


def already_created(person_id):
    """
    Логика тут такая - если в
    person_id есть '_' - значит
    это вручную сгенерированное значение
    вида person_uid_<уникальное значение>
    и такого пользователя нет нужды создавать
    так как он уже есть в базе

    :type person_id: str
    :rtype: bool
    """
    return '_' in person_id


def _get_check_key(person, attrs_to_check):
    return '_'.join(str(getattr(person, attr)) for attr in attrs_to_check)
