from django.conf import settings
from django.utils.deprecation import MiddlewareMixin
from django.contrib.auth import authenticate
from django.http import HttpResponseRedirect

from django_yauth.util import get_passport_url, get_yauth_type
from django_yauth.authentication_mechanisms.cookie import YandexUser

from intranet.audit.src.api_v1.views.view import APIView
from intranet.audit.src.api_v1.errors import YauthError
from .dao.user import get_user_by_request
from .models import User


class AuthMiddleware(MiddlewareMixin):
    def process_view(self, request, view_func, view_args, view_kwargs):
        view_class = getattr(view_func, 'view_class', None)
        need_yauser = self.check_exempt('need_yauser', (view_class, view_func))
        need_user = self.check_exempt('need_user', (view_class, view_func))

        if need_yauser:
            request.yauser = authenticate(request=request)
            if not request.yauser:
                if request.content_type == 'application/json':
                    view = APIView()
                    view.request = view.initialize_request(request)
                    view.headers = view.default_response_headers
                    response = view.handle_exception(YauthError('Invalid credentials'))
                    return view.finalize_response(view.request, response)
                else:
                    passport_url = get_passport_url('create',
                                                    get_yauth_type(request),
                                                    request=request,
                                                    retpath=True,
                                                    )

                    return HttpResponseRedirect(passport_url)

        if need_user:
            request.user = get_user_by_request(request=request)

    def check_exempt(self, attr, classes, ):
        return all(getattr(view, attr, True)
                   for view in classes)


class DevelopmentAuthMiddleware(MiddlewareMixin):
    def process_request(self, request):
        request.yauser = YandexUser(uid=settings.TEST_USER_UID, fields={'login': 'test'})
        request.user, _ = User.objects.using('default').get_or_create(**settings.TEST_USER_DATA)


def auth_middleware_exempt(need_yauser=False, need_user=False):
    if need_user and not need_yauser:
        raise AssertionError('You should set need_yauser=True if you need user')

    def wrapper(view_class):
        view_class.need_yauser = need_yauser
        view_class.need_user = need_user
        return view_class
    return wrapper
