from django.db import models
from django.contrib.auth.models import AbstractBaseUser, BaseUserManager
from django.utils.functional import cached_property
from django.contrib.auth.models import PermissionsMixin
from django.conf import settings

from model_utils.models import TimeStampedModel, SoftDeletableModel


VIEW_PERMISSION = {'core', 'files', 'advanced_filters', }


class UserManager(BaseUserManager):
    def create_user(self, login, uid, first_name, last_name, password=None):
        user = self.model(login=login, uid=uid,
                          first_name=first_name, last_name=last_name)
        user.set_password(password)
        user.save()
        return user

    def create_superuser(self, login, uid, first_name, last_name, password):
        user = self.create_user(login, uid, first_name, last_name, password=password)
        user.is_superuser = True
        user.save()
        return user


class User(AbstractBaseUser, PermissionsMixin):
    uid = models.CharField(max_length=16, primary_key=True,)
    login = models.CharField(max_length=50, unique=True,)
    first_name = models.CharField(max_length=120,)
    last_name = models.CharField(max_length=120,)
    is_active = models.BooleanField(default=True)
    is_staff = models.BooleanField(default=False)

    USERNAME_FIELD = 'login'

    REQUIRED_FIELDS = ['uid', 'first_name', 'last_name', ]

    objects = UserManager()

    def get_short_name(self):
        return self.login

    def get_full_name(self):
        return '"{}" "{}"'.format(self.last_name, self.first_name)

    def __str__(self):
        return self.alias

    @cached_property
    def alias(self):
        return '"{}" "{}" ({})'.format(self.first_name,
                                       self.last_name,
                                       self.login,
                                       )

    def has_module_perms(self, app_label):
        if app_label in VIEW_PERMISSION:
            return True
        return super().has_module_perms(app_label)


class StatedPerson(TimeStampedModel, SoftDeletableModel):
    """
    В этой модели хранятся данные пользователей как полученные
    со стаффа так и заведеные руками через админку "группы"
    (Совет Директоров, ...), руками заведенные записи
    можно отличить по GROUP_POSITION_SLUG в position

    Данные со стаффа фиксируются на момент сохранения
    модели и не обновляются в дальнейшем так как важно
    фиксировать данные пользователя (должность, группу, ...)
    на текущий момент.
    """
    GROUP_POSITION_SLUG = 'audit_group'

    id = models.CharField(max_length=50, primary_key=True, )
    uid = models.CharField(max_length=16, db_index=True, )
    login = models.CharField(max_length=50, )
    first_name = models.CharField(max_length=120, )
    last_name = models.CharField(max_length=120, )
    department = models.CharField(max_length=500, )
    department_slug = models.CharField(max_length=200, )
    position = models.CharField(max_length=200, )

    def __str__(self):
        return self.display_look

    @cached_property
    def format_item_display(self):
        department_url = settings.STAFF_DEPARTMENT_URL.format(self.department_slug)
        display_value = self.display_look
        return u'<a href="%s">%s</a>' % (department_url, display_value)

    @cached_property
    def display_look(self):
        if self.position == self.GROUP_POSITION_SLUG:
            return self.department
        template = '{} {} ({}) - {} - {}'
        return template.format(self.first_name,
                               self.last_name,
                               self.login,
                               self.position,
                               self.department,
                               )
