package main

import (
	"fmt"
	"net/http"
	"os"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"

	"a.yandex-team.ru/intranet/auth-checker/internal/cache"
	"a.yandex-team.ru/intranet/auth-checker/internal/database"
	"a.yandex-team.ru/intranet/auth-checker/internal/firewall"
	"a.yandex-team.ru/intranet/auth-checker/internal/handler"
	"a.yandex-team.ru/intranet/auth-checker/internal/staff"
)

type config struct {
	logLevel         log.Level
	staffEnvironment staff.Environment
}

func newConfig() config {
	if os.Getenv("DEPLOY_STAGE_ID") == "tools_auth-checker_production" {
		return config{
			logLevel:         log.InfoLevel,
			staffEnvironment: staff.Production,
		}
	} else {
		return config{
			logLevel:         log.DebugLevel,
			staffEnvironment: staff.Testing,
		}
	}
}

func main() {
	config := newConfig()

	logger, err := zap.NewDeployLogger(config.logLevel)
	if err != nil {
		fmt.Println(err)
		panic(err)
	}

	tvmClient, err := tvmtool.NewAnyClient(tvmtool.WithLogger(logger))
	if err != nil {
		logger.Fatal("unable to init tvm client, use tvmtool for development", log.Error(err))
	}

	bbClient, err := httpbb.NewIntranet(
		httpbb.WithLogger(logger),
		httpbb.WithTVM(tvmClient),
	)
	if err != nil {
		logger.Fatal("unable to init blackbox client", log.Error(err))
	}

	pgPinger := database.NewClient()

	dbClient, err := database.NewConnectionPools(logger, pgPinger)
	if err != nil {
		logger.Fatal("unable to init database client", log.Error(err))
	}

	ipChecker := firewall.NewIPChecker(
		logger,
		cache.NewSimpleCache(5*time.Minute),
		dbClient,
	)

	staffAPIClient := staff.NewClient(
		config.staffEnvironment,
		logger,
		cache.NewSimpleCache(60*time.Minute),
		tvmClient,
	)

	setLogger := handler.GetRequestLogger(logger)
	setTimeout := handler.GetRequestTimeout(logger, 60*time.Second)

	faceControlHandler := handler.GetFaceControlHandler(logger, bbClient, ipChecker, staffAPIClient)
	ipCheckerHandler := handler.GetIPCheckerHandler(logger, ipChecker)

	http.Handle("/get-staff/", setTimeout(setLogger(faceControlHandler)))
	http.Handle("/network_is_allowed", setTimeout(setLogger(ipCheckerHandler)))

	err = http.ListenAndServe(":80", nil)
	if err != nil {
		logger.Fatal("unable to start application")
	}
}
