package cache

import (
	"errors"
	"sync"
	"time"
)

type ValueWithDeadline struct {
	value    bool
	deadline time.Time
}

type SimpleCache struct {
	mutex   sync.RWMutex
	storage map[string]ValueWithDeadline
	ttl     time.Duration
}

func NewSimpleCache(ttl time.Duration) *SimpleCache {
	return &SimpleCache{
		storage: make(map[string]ValueWithDeadline),
		ttl:     ttl,
	}
}

func (c *SimpleCache) Get(key string) (bool, error) {
	c.mutex.RLock()
	defer c.mutex.RUnlock()

	value, ok := c.storage[key]
	if !ok {
		return false, errors.New("there is no value for key")
	}

	if value.deadline.Before(time.Now()) {
		return false, errors.New("value is expired")
	}

	return value.value, nil
}

func (c *SimpleCache) Set(key string, value bool) {
	c.mutex.Lock()
	defer c.mutex.Unlock()

	c.storage[key] = ValueWithDeadline{
		value:    value,
		deadline: time.Now().Add(c.ttl),
	}
}
