package database

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"os"
)

type PgPingerClient struct {
	httpClient *http.Client
	host       string
}

func NewClient() *PgPingerClient {
	host := os.Getenv("PGPINGER_URL")
	if host == "" {
		host = "http://localhost:9998"
	}
	return &PgPingerClient{
		httpClient: &http.Client{
			Transport: &http.Transport{
				MaxIdleConnsPerHost: 10,
			},
		},
		host: host,
	}
}

func (p *PgPingerClient) GetDatabaseHosts(ctx context.Context) ([]string, error) {
	request, err := http.NewRequestWithContext(ctx, "GET", p.host+"/hosts?preset=sync", nil)
	if err != nil {
		return nil, fmt.Errorf("unable to create request for pg-pinger: %w", err)
	}

	response, err := p.httpClient.Do(request)
	if err != nil {
		return nil, fmt.Errorf("unable to do pg-pinger request: %w", err)
	}
	defer response.Body.Close()

	var PingerResponse struct {
		Hosts []struct {
			Hostname string `json:"hostname"`
		} `json:"hosts"`
	}

	body, err := ioutil.ReadAll(response.Body)
	if err != nil {
		return nil, fmt.Errorf("unable to read pg-pinger response body: %w", err)
	}

	err = json.Unmarshal(body, &PingerResponse)
	if err != nil {
		return nil, fmt.Errorf("unable to parse pg-pinger response json: %w", err)
	}

	result := make([]string, len(PingerResponse.Hosts))
	for index, host := range PingerResponse.Hosts {
		result[index] = host.Hostname
	}

	return result, nil
}

func (p *PgPingerClient) GetHostsForReading(ctx context.Context) ([]string, error) {
	hosts, err := p.GetDatabaseHosts(ctx)
	if err != nil {
		return nil, err
	}

	return hosts, nil
}
