package database

import (
	"context"
	"fmt"
	"sync"
	"sync/atomic"

	"a.yandex-team.ru/library/go/core/log"
	"github.com/jackc/pgx/v4/pgxpool"
)

type HostManager interface {
	GetDatabaseHosts(ctx context.Context) ([]string, error)
	GetHostsForReading(ctx context.Context) ([]string, error)
}

type Client interface {
	GetReadConnection(ctx context.Context) (*pgxpool.Conn, error)
}

type ConnectionPools struct {
	hostManager HostManager
	pools       map[string]*pgxpool.Pool
	mutex       sync.RWMutex
}

func NewConnectionPools(logger log.Logger, hostManager HostManager) (*ConnectionPools, error) {
	c := ConnectionPools{
		hostManager: hostManager,
		pools:       make(map[string]*pgxpool.Pool),
	}

	hosts, err := hostManager.GetDatabaseHosts(context.Background())
	if err != nil {
		return nil, fmt.Errorf("unable to get hosts from manager: %w", err)
	}

	var errorsCount int32
	var wg sync.WaitGroup
	for _, host := range hosts {
		// TODO: если приложение поднимается при упавшем инстансе, то
		// горутина на создание пула должна висеть до его восстановления
		wg.Add(1)
		go func(host string) {
			defer wg.Done()

			hostField := log.String("host", host)
			logger.Info("trying to create database pool", hostField)

			config, err := pgxpool.ParseConfig("host=" + host)
			// TODO: кажется выключает prepared statements для работы с pgbouncer
			//       это не ок, поэтому следует разобраться в вопросе
			config.ConnConfig.PreferSimpleProtocol = true
			if err != nil {
				logger.Error("unable to parse database config", log.Error(err), hostField)
				atomic.AddInt32(&errorsCount, 1)
				return
			}

			pool, err := pgxpool.ConnectConfig(context.Background(), config)
			if err != nil {
				logger.Error("unable to create pool", log.Error(err), hostField)
				atomic.AddInt32(&errorsCount, 1)
				return
			}

			// TODO: закрывать pool?

			c.mutex.Lock()
			defer c.mutex.Unlock()
			c.pools[host] = pool

			logger.Info("database pool successfully created", hostField)
		}(host)
	}
	wg.Wait()

	if errorsCount == int32(len(hosts)) {
		return nil, fmt.Errorf("unable to create at least one pool for hosts %v", hosts)
	}

	return &c, nil
}

func (c *ConnectionPools) GetReadConnection(ctx context.Context) (*pgxpool.Conn, error) {
	hosts, err := c.hostManager.GetHostsForReading(ctx)
	if err != nil {
		return nil, err
	}

	c.mutex.RLock()
	defer c.mutex.RUnlock()

	for _, host := range hosts {
		pool, ok := c.pools[host]
		if !ok || pool == nil {
			continue
		}
		var conn *pgxpool.Conn
		conn, err = pool.Acquire(ctx)
		if err != nil {
			continue
		}
		return conn, nil
	}

	return nil, fmt.Errorf("unable to get database connection: %w", err)
}
