package handler

import (
	"context"
	"errors"
	"net/http"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/yandex/blackbox"

	"a.yandex-team.ru/intranet/auth-checker/internal/firewall"
	"a.yandex-team.ru/intranet/auth-checker/internal/staff"
)

const (
	statusAccepted = 380
	statusDenied   = 390
)

func GetFaceControlHandler(
	logger log.Logger,
	bbClient blackbox.Client,
	ipChecker *firewall.IPChecker,
	staffAPIClient *staff.StaffAPIClient,
) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		authChan := make(chan bool, 3)
		go func() {
			authChan <- isAuthWithSessionID(r.Context(), logger, bbClient, r)
		}()
		go func() {
			authChan <- isAuthWithOAuthToken(r.Context(), logger, bbClient, r)
		}()
		go func() {
			var isInternal bool
			var err error

			ip := getUserIP(r.Context())
			if ip != "" {
				isInternal, err = ipChecker.IsInternal(r.Context(), ip)
				if err != nil && !errors.Is(err, context.Canceled) {
					ctxlog.Error(r.Context(), logger, "unable to check ip", log.Error(err))
				}
			}

			authChan <- isInternal
		}()

		isDismissedChan := make(chan bool, 1)
		go func() {
			var isDismissed bool
			var err error

			login := getLoginFromPath(r.RequestURI)
			if login != "" {
				isDismissed, err = staffAPIClient.CheckIsDismissed(r.Context(), login)
				if err != nil {
					ctxlog.Error(r.Context(), logger, "unable to check login", log.Error(err))
				}
			}

			isDismissedChan <- isDismissed
		}()

		var authStatus bool
		for i := 0; i < 3; i++ {
			authStatus = <-authChan
			if authStatus {
				break
			}
		}

		if !authStatus {
			ctxlog.Warn(r.Context(), logger, "reason: failed to authenticate")
			w.WriteHeader(statusDenied)
			return
		}

		if <-isDismissedChan {
			ctxlog.Warn(r.Context(), logger, "reason: staff was dismissed without staff_agreement")
			w.WriteHeader(statusDenied)
			return
		}

		w.WriteHeader(statusAccepted)
	})
}

func getLoginFromPath(path string) string {
	// path pattern: /get-staff/<login>-(avatar|main).*
	avatarID := path[len("/get-staff/"):]
	slashIndex := strings.IndexByte(avatarID, '/')
	if slashIndex != -1 {
		avatarID = avatarID[:slashIndex]
	}
	if strings.HasSuffix(avatarID, "-main") {
		return avatarID[:len(avatarID)-len("-main")]
	} else if strings.HasSuffix(avatarID, "-avatar") {
		return avatarID[:len(avatarID)-len("-avatar")]
	} else {
		return avatarID[0:0]
	}
}

func isAuthWithSessionID(ctx context.Context, logger log.Logger, bbClient blackbox.Client, r *http.Request) bool {
	ctxlog.Debug(ctx, logger, "cookie: trying to auth")
	sessionIDCookie, _ := r.Cookie("Session_id")
	if sessionIDCookie == nil {
		ctxlog.Debug(ctx, logger, "cookie: empty cookie")
		return false
	}

	request := blackbox.SessionIDRequest{
		SessionID: sessionIDCookie.Value,
		UserIP:    getUserIP(r.Context()),
		Host:      r.Host,
	}

	_, err := bbClient.SessionID(ctx, request)
	if err != nil {
		if !errors.Is(err, context.Canceled) {
			ctxlog.Error(ctx, logger, "unable to check session_id", log.Error(err))
		}
		return false
	}

	return true
}

func isAuthWithOAuthToken(ctx context.Context, logger log.Logger, bbClient blackbox.Client, r *http.Request) bool {
	ctxlog.Debug(ctx, logger, "oauth: trying to auth")
	authorizationHeader := r.Header.Get("Authorization")
	oauthTokenPrefix := "OAuth "

	if authorizationHeader == "" || !strings.HasPrefix(authorizationHeader, oauthTokenPrefix) {
		ctxlog.Debug(ctx, logger, "oauth: empty token")
		return false
	}

	request := blackbox.OAuthRequest{
		OAuthToken: authorizationHeader[len(oauthTokenPrefix):],
		UserIP:     getUserIP(r.Context()),
	}

	_, err := bbClient.OAuth(ctx, request)
	if err != nil {
		if !errors.Is(err, context.Canceled) {
			ctxlog.Error(ctx, logger, "unable to check oauth", log.Error(err))
		}
		return false
	}

	return true
}
