package staff

import (
	"context"
	"fmt"
	"net/http"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/yandex/tvm"

	"a.yandex-team.ru/intranet/auth-checker/internal/cache"
)

type Environment struct {
	host string
}

var (
	Production = Environment{host: "https://staff-api.yandex-team.ru"}
	Testing    = Environment{host: "https://staff-api.test.yandex-team.ru"}
)

type StaffAPIClient struct {
	httpClient *http.Client
	host       string
	cache      cache.Cache
	tvmClient  tvm.Client
	logger     log.Logger
}

func NewClient(env Environment, logger log.Logger, cache cache.Cache, tvmClient tvm.Client) *StaffAPIClient {
	client := StaffAPIClient{
		httpClient: &http.Client{
			Transport: &http.Transport{
				MaxIdleConnsPerHost: 100,
			},
		},
		host:      env.host,
		cache:     cache,
		tvmClient: tvmClient,
		logger:    logger,
	}
	return &client
}

func (s *StaffAPIClient) CheckIsDismissed(ctx context.Context, login string) (bool, error) {
	ctxlog.Debugf(ctx, s.logger, "staff-api: trying to check login: %v", login)
	if s.cache != nil {
		value, err := s.cache.Get(login)
		if err == nil {
			ctxlog.Debugf(ctx, s.logger, "staff-api: using cache value: %v", value)
			return value, nil
		}
	}

	tvmToken, err := s.tvmClient.GetServiceTicketForAlias(ctx, "staff-api")
	if err != nil {
		return false, fmt.Errorf("unable to get tvm ticket for staff-api request: %w", err)
	}

	request, err := http.NewRequestWithContext(ctx, "POST", s.host+"/v3/persons", nil)
	if err != nil {
		return false, fmt.Errorf("unable to create request for staff-api: %w", err)
	}

	request.Header.Add("X-Ya-Service-Ticket", tvmToken)

	params := request.URL.Query()
	params.Add("_fields", "id")
	params.Add("_one", "1")
	params.Add("login", login)
	params.Add("official.is_dismissed", "true")
	params.Add("official.staff_agreement", "false")
	params.Add("memorial", "null")
	request.URL.RawQuery = params.Encode()

	response, err := s.httpClient.Do(request)
	if err != nil {
		return false, fmt.Errorf("unable to do staff-api request: %w", err)
	}
	_ = response.Body.Close()

	ctxlog.Debugf(ctx, s.logger, "staff-api: status code %v for query: %v", response.StatusCode, request.URL.RawQuery)
	if response.StatusCode == http.StatusOK {
		s.cache.Set(login, true)
		return true, nil
	} else {
		s.cache.Set(login, false)
		return false, nil
	}
}
