# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json
import unicodedata

from django import forms
from django.conf import settings
from django.core.exceptions import ValidationError
from django.forms.models import ModelFormMetaclass
from django.template.defaultfilters import filesizeformat
from django.utils import timezone
from django.utils.translation import gettext_lazy as _

from core import validators
from core.models import PaymentInfo, NewPaymentInfo, Reporter, Reward
from core.utils.balance import BalanceClient

import six


class YandexFormMixin(object):

    def __init__(self, *args, **kwargs):
        super(YandexFormMixin, self).__init__(*args, **kwargs)
        self._init_widgets()

    def _init_widgets(self):
        for field in self.fields.itervalues():
            if isinstance(field.widget, forms.TextInput):
                field.widget.attrs = {
                    'size': '1',
                    'autocomplete': 'off',
                    'placeholder': '',
                    'class': 'input__input',
                }
            elif isinstance(field.widget, forms.Textarea):
                field.widget.attrs = {
                    'rows': '9',
                    'autocomplete': 'off',
                    'placeholder': '',
                    'class': 'textarea__textarea',
                }


class ReporterForm(YandexFormMixin, forms.ModelForm):

    show_on_hall = forms.BooleanField(
        widget=forms.CheckboxInput(attrs={'autocomplete': 'off'}),
        required=False,
    )

    def __init__(self, *args, **kwargs):
        super(ReporterForm, self).__init__(*args, **kwargs)
        self.fields['twitter_link'].widget.attrs['placeholder'] = _('Twitter login')
        self.fields['fb_link'].widget.attrs['placeholder'] = _('Facebook login')
        self.fields['hacker_one_link'].widget.attrs['placeholder'] = _('HackerOne login')
        self.fields['bug_crowd_link'].widget.attrs['placeholder'] = _('BugCrowd login')
        self.fields['personal_website'].widget.attrs['placeholder'] = _('http://example.com')
        self.fields['contact_email'].widget.attrs['pattern'] = r'[a-z0-9._+-]+@[a-z0-9.-]+\.[a-z]{2,4}$'

    class Meta:
        model = Reporter
        fields = [
            'display_name',
            'contact_email',
            'fb_link',
            'twitter_link',
            'hacker_one_link',
            'bug_crowd_link',
            'personal_website',
            'company_name',
            'show_on_hall',
        ]


class PaymentForm(YandexFormMixin, forms.ModelForm):
    # TODO(remedy) Merge with external app.forms.PaymentInfoForm

    document = forms.FileField(required=False)
    is_russian_resident = forms.BooleanField(
        widget=forms.CheckboxInput(attrs={'autocomplete': 'off'}),
        required=False,
    )

    class Meta:
        model = PaymentInfo
        fields = [
            'ru_fio',
            'ru_account_number',
            'ru_bank_name',
            'ru_bic_code',
            'ru_correspond_number',
            'account_holder',
            'account_number',
            'iban',
            'bank_name',
            'swift_code',
            'is_russian_resident',
            'is_actual',
            'mdm_ticket',
        ]


class NewPaymentInfoFormMetaclass(ModelFormMetaclass):
    def __new__(cls, name, bases, attrs):
        base_metas = [base.Meta for base in bases if hasattr(base, 'Meta')]
        new_meta = attrs.get('Meta')
        if not new_meta and not base_metas:
            raise ValueError('New class should have Meta class attribute defined')
        meta = new_meta or base_metas[0]
        assert meta.model == NewPaymentInfo
        datasync_fields = NewPaymentInfo.DATASYNC_FIELDS
        for fieldname in datasync_fields:
            fixed_fieldname = fieldname.replace('-', '_')
            in_base_attrs = bool([base for base in bases if fixed_fieldname in getattr(base, 'base_fields', {})])
            if fixed_fieldname not in attrs and not in_base_attrs:
                if fixed_fieldname == 'type':
                    continue  # поле is_russian_resident будет преобразовано в type при сохранении
                attrs[fixed_fieldname] = forms.CharField(required=False)
        return super(NewPaymentInfoFormMetaclass, cls).__new__(cls, name, bases, attrs)


class NewPaymentInfoForm(YandexFormMixin, forms.ModelForm):
    __metaclass__ = NewPaymentInfoFormMetaclass

    IGNORED_MODEL_FIELDS = {'type'}

    RESIDENT_BANK_TYPES = (
        ('1', _('payment_info.bank_type.sberbank')),
        ('2', _('payment_info.bank_type.other_bank')),
        # https://st.yandex-team.ru/BUGBOUNTY-461
        # ('0', _('payment_info.bank_type.not_provided')),
        # ('3', _('payment_info.bank_type.yamoney')),
        # ('7', _('payment_info.bank_type.payoneer')),
    )

    is_russian_resident = forms.BooleanField(required=False)
    email = forms.EmailField(required=False)
    birthday = forms.DateField(required=False)
    passport_d = forms.DateField(required=False)
    bank_type = forms.ChoiceField(choices=RESIDENT_BANK_TYPES, required=False, initial='1', widget=forms.RadioSelect)
    account = forms.CharField(max_length=34, required=False,
                              validators=[validators.IBANValidator()])  # или всё же iban?
    swift = forms.CharField(max_length=13, required=False, validators=[validators.SWIFTValidator()])
    bik = forms.CharField(max_length=9, required=False, validators=[validators.BICValidator()])
    pfr = forms.CharField(max_length=14, required=False, validators=[validators.SNILSValidator()])
    document = forms.FileField(max_length=64, required=False)  # max_length – длина имени файла
    account_type = forms.CharField(max_length=16, required=False)

    @property
    def payment_via_bank(self):
        # Сбербанк или любой другой банк
        # Описание здесь: https://wiki.yandex-team.ru/Balance/XmlRpc/#objazatelnyepoljaxjeshavzavisimostiottype
        return self.initial['bank_type'] in ('1', '2')

    class Meta:
        model = NewPaymentInfo
        exclude = [
            'reporter',
        ]

    def __init__(self, *args, **kwargs):
        super(NewPaymentInfoForm, self).__init__(*args, **kwargs)
        self._init_placeholders()
        self._set_already_filled_fields()

    def _set_already_filled_fields(self):
        self.filled_fields = set()
        self.filled_latin_fields = set()

        if not self.instance:
            return

        if self.instance.datasync_values.get('document'):
            self.filled_fields.add('document')
            self.filled_latin_fields.add('document')

    def _init_placeholders(self):
        placeholders = {
            'account': _('iban.iban_format'),
            'swift': _('swift.swift_format'),
            'birthday': _('payment_info.date_format'),
            'passport_d': _('payment_info.date_format'),
            'pfr': _('payment_info.pfr_format'),
            'bik': _('payment_info.bic_format'),
        }
        for name, placeholder in placeholders.items():
            self.fields[name].widget.attrs['placeholder'] = placeholder

    def clean_document(self):
        document = self.cleaned_data.get('document')
        if not document:
            return
        size = document.size
        if size > settings.MAX_ATTACHMENT_SIZE:
            limit_string = filesizeformat(settings.MAX_ATTACHMENT_SIZE)
            limit_string = unicodedata.normalize('NFKD', limit_string)
            raise ValidationError('File size cannot be more than {}'.format(limit_string))
        return document

    def clean(self):
        super(NewPaymentInfoForm, self).clean()
        self._clean_captcha(self.cleaned_data)
        if self.errors:
            return self.cleaned_data
        if self.cleaned_data['is_russian_resident']:
            self._resident_required_fields()
        else:
            self._nonresident_required_fields()

        self._remove_empty_fields()

        # в старой модели и в нашей логике резиденство – булев флаг, а в Балансе – это тип клиента (enumeration)
        if self.cleaned_data['is_russian_resident']:
            self.cleaned_data['type'] = NewPaymentInfo.RESIDENT_TYPE
        else:
            self.cleaned_data['type'] = NewPaymentInfo.NONRESIDENT_TYPE

        del self.cleaned_data['is_russian_resident']

        for date_field in ['birthday', 'passport_d']:
            if self.cleaned_data.get(date_field):
                self.cleaned_data[date_field] = self.cleaned_data[date_field].strftime('%Y-%m-%d')

        self._simulate_create_person()
        return self.cleaned_data

    def _clean_captcha(self, cleaned_data):
        pass  # для наследуемых классов, вызывается в clean()

    def _resident_required_fields(self):
        required_fields = {
            field.replace('-', '_')
            for field
            in (NewPaymentInfo.RESIDENT_REQUIRED_FIELDS - self.IGNORED_MODEL_FIELDS)
        }
        required_fields -= self.filled_fields

        bank_type = self.cleaned_data.get('bank_type') or self.initial['bank_type']
        subfields = NewPaymentInfo.BANK_TYPE_REQUIRED_FIELDS[bank_type]
        for subfield in subfields:
            subfield = subfield.replace('-', '_')
            if subfield not in self.filled_fields:
                required_fields.add(subfield)

        for field_name in required_fields:
            self._check_required_field(field_name)

    def _nonresident_required_fields(self):
        required_fields = {
            field.replace('-', '_')
            for field
            in (NewPaymentInfo.NONRESIDENT_REQUIRED_FIELDS - self.IGNORED_MODEL_FIELDS)
        }

        account_type = self.cleaned_data.get('account_type')
        if account_type == 'account':
            required_fields.add('account')
        elif account_type == 'ben_account':
            required_fields.add('ben_account')
        else:
            # Если не заполнено ни одно account-поле, то account_type обязателен
            if not any(f in self.filled_latin_fields for f in ('account', 'ben_account')):
                self._check_required_field('account_type')

        for field_name in required_fields:
            if field_name not in self.filled_latin_fields:
                self._check_required_field(field_name)
            self._check_latin_content(field_name)

    def _check_required_field(self, field_name):
        field = self.fields.get(field_name)
        value = self.cleaned_data.get(field_name)
        if value in field.empty_values:
            message = field.error_messages['required']
            error = ValidationError(message, code='required')
            self.add_error(field_name, error)

    def _check_latin_content(self, field_name):
        value = self.cleaned_data.get(field_name)
        if not isinstance(value, six.text_type):
            return
        if not all([ord(ch) < 128 for ch in value]):
            message = 'The field must contain only latin characters'
            error = ValidationError(message, code='latin')
            self.add_error(field_name, error)

    def _remove_empty_fields(self):
        for field_name in self.cleaned_data.keys():
            field = self.fields.get(field_name)
            value = self.cleaned_data.get(field_name)
            if value in field.empty_values:
                del self.cleaned_data[field_name]

    def _simulate_create_person(self):
        client = BalanceClient()
        potential_datasync_values = self.instance.datasync_values.copy()
        potential_datasync_values.update(self.cleaned_data)
        old_keys = potential_datasync_values.keys()
        for key in old_keys:
            datasync_key = key.replace('_', '-')
            if datasync_key != key:
                potential_datasync_values[datasync_key] = potential_datasync_values[key]
                del potential_datasync_values[key]

        params = client.get_person_params(potential_datasync_values, settings.BALANCE_SIMULATION_CLIENT_ID)
        params['simulate'] = '1'
        for _ in range(settings.BALANCE_SIMULATION_ATTEMPTS):
            try:
                balance_response = client.send_data((settings.BALANCE_MANAGER_UID, params), 'Balance.CreatePerson')
                response_json = balance_response[0][0]
                errors = response_json['errors']
                for error in errors:
                    self.add_error(error['field'].lower(), error['description'])
                self.instance.balance_simulation_result = json.dumps(response_json)
                break
            except Exception as e:
                # Пропускаем исключения от баланса
                self.instance.balance_simulation_result = unicode(e)

    def _update_rewards_statuses(self):
        reporter = self.instance.reporter
        reporter.rewards.filter(status=Reward.ST_BAD_DETAILS).update(status=Reward.ST_NEW)

    def save(self, commit=True):
        update_document_timestamp = bool(self.cleaned_data.get('document'))
        if update_document_timestamp:
            self.instance.document_uploaded_at = timezone.now()
        self.instance.datasync_values.update(self.cleaned_data)

        account_type = self.cleaned_data.get('account_type')
        if account_type == 'account':
            self.instance.datasync_values['ben-account'] = ''
        elif account_type == 'ben_account':
            self.instance.datasync_values['account'] = ''

        old_keys = self.instance.datasync_values.keys()
        for key in old_keys:
            datasync_key = key.replace('_', '-')
            if datasync_key != key:
                self.instance.datasync_values[datasync_key] = self.instance.datasync_values[key]
                del self.instance.datasync_values[key]

        self._update_rewards_statuses()
        return super(NewPaymentInfoForm, self).save(commit=commit)
