# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from core import exceptions
from core import triggers
from core.utils import dates


__all__ = ['Badge', 'UserBadge']

logger = logging.getLogger(__name__)


@python_2_unicode_compatible
class Badge(models.Model):
    CI_DAY = 0
    CI_MONTH = 1
    CI_QUARTER = 2

    CHECK_INTERVALS = (
        # Translators: EN Daily
        (CI_DAY, _('badge.ci_day')),
        # Translators: EN Monthly
        (CI_MONTH, _('badge.ci_month')),
        # Translators: EN every quarter
        (CI_QUARTER, _('badge.ci_quarter')),
    )

    name = models.CharField(max_length=255)
    name_en = models.CharField(max_length=255)
    image_url = models.URLField()
    is_automatic = models.BooleanField(default=False)
    is_active = models.BooleanField(default=True)
    is_exclusive = models.BooleanField(default=False)

    trigger = models.IntegerField(
        choices=triggers.registry.registered(),
        null=True, default=None, blank=True,
    )

    check_interval = models.IntegerField(
        choices=CHECK_INTERVALS, null=True, default=CI_QUARTER, blank=True,
    )

    def __str__(self):
        return self.name_en
    
    @property
    def active_holders(self):
        return self.userbadge_set.filter(deactivated__isnull=True)

    @property
    def inactive_holders(self):
        return self.userbadge_set.exclude(deactivated__isnull=True)

    @property
    def template_code(self):
        """Code for MailTemplate."""
        return 'badge.%s' % self.id

    @property
    def deprive_template_code(self):
        """Code for MailTemplate."""
        return 'badge_deprive.%s' % self.id

    def assign_by_trigger(self):
        if self.can_be_auto_assigned():
            try:
                trigger = triggers.registry.get_trigger(self.trigger)
            except triggers.TriggerNotFound:
                logger.warn('%s: trigger not found', self)
            else:
                trigger(self.id)

    def can_be_auto_assigned(self):
        return self.is_active and self.is_automatic

    def award(self, reporter):
        UserBadge.objects.create(user=reporter, badge=self)
        
    def exclusive_holder_changes(self, reporter):
        if self.is_exclusive:
            if self.active_holders.all():
                awarded_reporter = self.active_holders.get().user
                return reporter != awarded_reporter
        return False

    def deprive_exclusive_holder(self):
        if self.is_exclusive:
            self.active_holders.get().deactivate()

    def is_not_assigned(self, reporter):
        status = self.active_holders.filter(user=reporter).exists()
        logger.debug('Badge is assigned: %s', status)
        return not self.active_holders.filter(user=reporter).exists()


class ActiveBadgeHoldersManager(models.Manager):

    def get_queryset(self):
        queryset = super(ActiveBadgeHoldersManager, self).get_queryset()
        return queryset.filter(deactivated__isnull=True)


@python_2_unicode_compatible
class UserBadge(models.Model):
    user = models.ForeignKey('core.Reporter')
    badge = models.ForeignKey(Badge)
    activated = models.DateTimeField(auto_now_add=True)
    deactivated = models.DateTimeField(null=True, default=None)

    objects = models.Manager()
    active = ActiveBadgeHoldersManager()

    def __str__(self):
        return self.badge.name_en

    def save(self, *args, **kwargs):
        if self.badge.is_exclusive and self._badge_already_taken():
            raise exceptions.ModelException(
                'Exclusive badge already taken')
        elif self._badge_already_taken_by_user():
            raise exceptions.ModelException(
                'Badge already taken by %s' % self.user)
        return super(UserBadge, self).save(*args, **kwargs)

    def _badge_already_taken(self):
        return (UserBadge.active
                .filter(badge=self.badge)
                .exclude(user=self.user)
                .exists())

    def _badge_already_taken_by_user(self):
        return (UserBadge.active
                .exclude(pk=self.pk)
                .filter(badge=self.badge, user=self.user)
                .exists())

    def deactivate(self):
        if self.deactivated is None:
            self.deactivated = dates.today()
            self.save()
            self.notify_user(deprive=True)

    def notify_user(self, deprive=False):
        from core.utils import mail
        try:
            mail.BadgeMessage(self, deprive=deprive).send()
        except mail.BadgeMessageError:
            pass
