from datetime import date

from django.db import models
from django.template.defaultfilters import date as _date
from django.utils.encoding import python_2_unicode_compatible


__all__ = ['List', 'Item', 'HallOfFameGroup']


class PublishedManager(models.Manager):

    def get_queryset(self):
        return super(PublishedManager, self).get_queryset().filter(is_published=True).order_by('year', 'month')

    def _last_or_today(self):
        last = self.last()
        if not last:
            last = date.today()
        return last

    def last_month(self):
        return self._last_or_today().month

    def last_year(self):
        return self._last_or_today().year

    def years(self):
        years = set(self.values_list('year', flat=True))
        if not years:
            years = [date.today().year]
        return list(sorted(years, reverse=True))


@python_2_unicode_compatible
class List(models.Model):

    class Meta:
        app_label = 'core'

    month = models.SmallIntegerField()
    year = models.SmallIntegerField()
    is_published = models.BooleanField(default=False)
    members = models.ManyToManyField('core.Reporter',
                                     related_name='hall_of_fame_groups',
                                     through='Item',
                                     through_fields=('list', 'user'))

    class Meta:
        unique_together = ('month', 'year')

    def __str__(self):
        return _date(date(self.year, self.month, 1), 'F Y')


# TODO(remedy) This model should replace List and Item in the future.
class HallOfFameGroup(List):

    objects = models.Manager()
    published = PublishedManager()

    class Meta:
        proxy = True

    def add_member(self, member):
        _, created = Item.objects.get_or_create(list=self, user=member)
        return created

    def add_members(self, members):
        added = 0
        for member in members:
            added += int(self.add_member(member))
        return added

    def publish(self):
        if self.members.exists():
            self.is_published = True
            self.save()


@python_2_unicode_compatible
class Item(models.Model):

    class Meta:
        app_label = 'core'

    list = models.ForeignKey(List, related_name='items')
    user = models.ForeignKey('core.Reporter', on_delete=models.CASCADE)

    def __str__(self):
        return '%s in %s/%s' % (self.user.login, self.list.year, self.list.month)
