# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

from django.core.urlresolvers import reverse
from django.utils.translation import ugettext_lazy as _
from django.db import models
from django.db.models.signals import post_save
from markdown import markdown
from markdown.extensions import extra

__all__ = ['StaticPage', 'EditablePage', 'PageTranslation']

logger = logging.getLogger(__name__)

class StaticPage(models.Model):

    class Meta:
        app_label = 'core'
        unique_together = ('language', 'page_type')


    LANGUAGE_RUSSIAN = 0
    LANGUAGE_ENGLISH = 1

    LANGUAGES = (
        # Translators: EN russian
        (LANGUAGE_RUSSIAN, _('language.russian')),
        # Translators: EN english
        (LANGUAGE_ENGLISH, _('language.english')),
    )

    STATUS_DRAFT = 0
    STATUS_IN_TANKER = 1
    STATUS_PUBLISHED = 3

    STATUSES = (
        # Translators: EN published
        (STATUS_PUBLISHED, _('page.published')),
        # Translators: EN translating
        (STATUS_IN_TANKER, _('page.in_tanker')),
        # Translators: EN draft
        (STATUS_DRAFT, _('page.draft')),
    )

    TYPE_INDEX = 'index'
    TYPE_FAQ = 'faq'
    TYPE_REPORT = 'report'
    TYPE_HALL_OF_FAME = 'hall_of_fame'

    PAGE_TYPE = (
        (TYPE_INDEX, TYPE_INDEX),
        (TYPE_FAQ, TYPE_FAQ),
        (TYPE_REPORT, TYPE_REPORT),
        (TYPE_HALL_OF_FAME, TYPE_HALL_OF_FAME),
    )

    TRANSLATABLE_FIELDS = {'text', 'name'}
    TANKER_TYPE = 'pages'

    page_type = models.CharField(max_length=100, choices=PAGE_TYPE, default=TYPE_INDEX)
    name = models.CharField(max_length=255,  default="")
    status = models.IntegerField(choices=STATUSES, default=STATUS_DRAFT)
    text = models.TextField(default="")
    formatted_text = models.TextField(default="")
    language = models.IntegerField(choices=LANGUAGES, default=LANGUAGE_RUSSIAN)
    created = models.DateTimeField(auto_now_add=True)
    modified = models.DateTimeField(auto_now=True)

    def save(self, *args, **kwargs):
        if self.status == self.STATUS_PUBLISHED:
            self.formatted_text = markdown(
                self.text, extensions=[extra.ExtraExtension()])
        super(StaticPage, self).save(*args, **kwargs)

    def get_tanker_language_code(self):
        if self.language == self.LANGUAGE_RUSSIAN:
            return 'ru'
        elif self.language == self.LANGUAGE_ENGLISH:
            return 'en'
        raise Exception('Forgotten Language')

    @classmethod
    def get_language_code(cls, language):
        if language == 'ru':
            return cls.LANGUAGE_RUSSIAN
        if language == 'en':
            return cls.LANGUAGE_ENGLISH
        logger.critical("No language code for language: %s" % language)
        return cls.LANGUAGE_ENGLISH


class EditablePage(models.Model):
    SLUG_MAXLEN = 16
    slug = models.SlugField(max_length=SLUG_MAXLEN, unique=True)
    order = models.IntegerField()

    @property
    def is_removable(self):
        return self.order > 0


class PublishedManager(models.Manager):

    def get_queryset(self):
        return super(PublishedManager, self).get_queryset().filter(
            status=PageTranslation.STATUS_PUBLISHED)


class PageTranslation(models.Model):
    TITLE_MAXLEN = 255
    STATUS_DRAFT = 'draft'
    STATUS_UPLOADED = 'uploaded'
    STATUS_DOWNLOADED = 'downloaded'
    STATUS_PUBLISHED = 'published'
    STATUSES = (
        (STATUS_DRAFT, _('page.draft')),
        (STATUS_UPLOADED, _('page.uploaded')),
        (STATUS_DOWNLOADED, _('page.downloaded')),
        (STATUS_PUBLISHED, _('page.published')),
    )

    page = models.ForeignKey('EditablePage')
    status = models.CharField(max_length=16, choices=STATUSES)
    title_ru = models.CharField(max_length=TITLE_MAXLEN)
    text_ru = models.TextField()
    html_ru = models.TextField()
    title_en = models.CharField(max_length=TITLE_MAXLEN, blank=True, null=True)
    text_en = models.TextField(blank=True, null=True)
    html_en = models.TextField(blank=True, null=True)
    created = models.DateTimeField(auto_now_add=True)
    modified = models.DateTimeField(auto_now=True)

    objects = models.Manager()
    published = PublishedManager()

    class Meta:
        unique_together = ('page', 'status')

    def get_absolute_url(self):
        return reverse('page:update-%s' % self.status, kwargs={'slug': self.page.slug})

    def save(self, *args, **kwargs):
        if self.text_ru is not None:
            self.html_ru = markdown(
                self.text_ru, extensions=[extra.ExtraExtension()])
        if self.text_en is not None:
            self.html_en = markdown(
                self.text_en, extensions=[extra.ExtraExtension()])
        super(PageTranslation, self).save(*args, **kwargs)

    @property
    def page_slug(self):
        return self.page.slug


def create_page_with_other_language(sender, instance, **kwargs):
    if instance.language == StaticPage.LANGUAGE_RUSSIAN:
        new_language = StaticPage.LANGUAGE_ENGLISH
    else:
        new_language = StaticPage.LANGUAGE_RUSSIAN
    obj, created = StaticPage.objects.get_or_create(page_type=instance.page_type, 
                                                    language=new_language, 
                                                    defaults={'name':instance.name})


post_save.connect(create_page_with_other_language, sender=StaticPage)
