# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

from cryptography.fernet import Fernet
from io import BytesIO
import transliterate
from unidecode import unidecode

from django.contrib.auth.models import AbstractBaseUser, AbstractUser, BaseUserManager, UserManager
from django.core.urlresolvers import reverse
from django.core.validators import RegexValidator
from django.db import models
from django.utils.encoding import python_2_unicode_compatible, force_bytes
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from core.utils import datasync
from core.utils import external_userinfo_by_uid, BlackboxUserError, BlackboxUser
from core.utils import language as language_utils


__all__ = ['Reporter', 'Administrator', 'PaymentInfo', 'NewPaymentInfo', 'Document', 'Responsible']

logger = logging.getLogger(__name__)


class ReporterManager(BaseUserManager):

    def create_user(self, username, password):
        user = self.model(uid=username)
        user.set_password(password)
        user.save(using=self._db)
        return user

    def create_superuser(self, username, password):
        logger.error('External db should be created without superuser')
        raise Exception('External db should be created without superuser')


class AdministratorManager(UserManager):
    pass


@python_2_unicode_compatible
class Reporter(AbstractBaseUser):
    username = models.CharField(max_length=255)
    uid = models.BigIntegerField(unique=True)
    email = models.EmailField(blank=True)

    # TODO(remedy) These fields should be placed in separate profile.
    display_name = models.CharField(max_length=255, blank=True)
    contact_name = models.CharField(max_length=255, blank=True)
    contact_email = models.EmailField(blank=True)
    fb_link = models.CharField(
        max_length=255, blank=True,
        validators=[RegexValidator(r'^[A-Za-z\d.]{5,}$')])
    twitter_link = models.CharField(
        max_length=255, blank=True,
        validators=[RegexValidator(r'^[A-Za-z\d_]{1,15}$')])
    hacker_one_link = models.CharField(
        max_length=255, blank=True,
        validators=[RegexValidator(r'^[A-Za-z\d_-]{1,25}$')])
    bug_crowd_link = models.CharField(
        max_length=255, blank=True,
        validators=[RegexValidator(r'^[A-Za-z\d_-]*$')])
    personal_website = models.URLField(blank=True)
    address = models.TextField(blank=True)
    company_name = models.CharField(max_length=255, blank=True)
    about = models.TextField(blank=True)
    show_on_hall = models.BooleanField(default=True)
    points = models.IntegerField(default=0)
    badges = models.ManyToManyField(
        'Badge', related_name='holders', through='UserBadge')
    balance_client_id = models.TextField(null=True, blank=True)
    balance_person_id = models.TextField(null=True, blank=True)
    balance_nonresident_person_id = models.TextField(null=True, blank=True)
    balance_contract_id = models.TextField(null=True, blank=True)

    USERNAME_FIELD = 'uid'

    class Meta:
        app_label = 'core'

    objects = ReporterManager()

    def __str__(self):
        return 'Uid {0.uid}'.format(self)

    def get_absolute_url(self):
        return reverse('user_profile:edit', kwargs={'pk': self.id})

    def has_contacts(self):
        if self.company_name or self.contact_email:
            return True
        return False

    def has_social_links(self):
        if self.twitter_link or self.fb_link:
            return True
        return False

    @cached_property
    def blackbox_user(self):
        try:
            return external_userinfo_by_uid(self.uid)
        except BlackboxUserError:
            return BlackboxUser(uid=self.uid)

    @property
    def login(self):
        return self.blackbox_user.username

    @property
    def default_email(self):
        return self.blackbox_user.email

    @property
    def is_russian_resident(self):
        try:
            if 'type' in self.new_payment_info.datasync_values:
                return (self.new_payment_info.datasync_values['type'] == self.new_payment_info.RESIDENT_TYPE)
        except (AttributeError, NewPaymentInfo.DoesNotExist):
            pass
        try:
            return self.payment_info.is_russian_resident
        except (AttributeError, PaymentInfo.DoesNotExist):
            pass
        return None


@python_2_unicode_compatible
class PaymentInfo(models.Model):
    reporter = models.OneToOneField(Reporter, related_name='payment_info')
    last_modified = models.DateTimeField(auto_now=True, null=True)

    account_holder = models.CharField(max_length=255, blank=True)
    account_number = models.CharField(max_length=255, blank=True)

    iban = models.CharField(max_length=34, blank=True)
    bank_name = models.CharField(max_length=255, blank=True)
    swift_code = models.CharField(max_length=50, blank=True)

    ru_fio = models.CharField(max_length=255, blank=True)
    ru_account_number = models.CharField(max_length=50, blank=True)
    ru_bank_name = models.CharField(max_length=255, blank=True)
    ru_bic_code = models.CharField(max_length=50, blank=True)
    ru_correspond_number = models.CharField(max_length=50, blank=True)

    is_russian_resident = models.BooleanField(default=False)

    document_uploaded_time = models.DateTimeField(blank=True, null=True)
    document_name = models.CharField(max_length=255, blank=True, null=True)

    mdm_ticket = models.URLField(blank=True)

    is_actual = models.BooleanField(default=False)

    class Meta:
        app_label = 'core'
        verbose_name_plural = 'payment info profiles'

    def __str__(self):
        return str(self.reporter)

    def get_absolute_url(self):
        return reverse('user_profile:edit_payment', kwargs={'pk': self.reporter.pk})

    def get_actual_document_id(self):
        try:
            document = self.document_set.latest('created')
        except Document.DoesNotExist:
            return None
        return document.document_id

    def get_all_documents(self):
        documents = self.document_set.all()
        return documents

    @property
    def account_holder_cyrillic(self):
        if language_utils.is_cyrillic_only(self.account_holder):
            return self.account_holder
        else:
            cyrillic_name = transliterate.translit(
                unidecode(self.account_holder), 'ru')
            return '%s\n(%s)' % (cyrillic_name, self.account_holder)


@python_2_unicode_compatible
class NewPaymentInfo(models.Model):
    # Поля в целом соответствуют вот этому списку:
    # https://wiki.yandex-team.ru/Balance/XmlRpc/#objazatelnyepoljaxjeshavzavisimostiottype
    DATASYNC_FIELDS = {
        'type',
        'lname',
        'fname',
        'mname',
        'phone',
        'email',
        'pfr',
        'inn',
        'legal-address-postcode',
        'legal-address-gni',
        'legal-address-region',
        'legal-address-city',
        'legal-address-street',
        'legal-address-home',
        'legal-fias-guid',
        'legaladdress',
        'address-gni',
        'address-region',
        'address-postcode',
        'address-code',
        'birthday',
        'passport-d',
        'passport-s',
        'passport-n',
        'passport-e',
        'passport-code',
        'bank-type',
        'person-account',
        'yamoney-wallet',
        'payoneer-wallet',
        'city',
        'country',
        'postcode',
        'postaddress',
        'ben-account',
        'ben-bank',
        'bik',
        'account',
        'swift',
        'document',
        'bank-correspondent-name',
        'bank-correspondent-account',
    }
    RESIDENT_REQUIRED_FIELDS = {
        'type',
        'lname',
        'fname',
        'mname',
        'phone',
        'email',
        'pfr',
        'inn',
        'legal-address-postcode',
        'legal-address-gni',
        'legal-address-region',
        'legal-address-city',
        'legal-address-street',
        'legal-address-home',
        'legal-fias-guid',
        'address-gni',
        'address-region',
        'address-postcode',
        'address-code',
        'birthday',
        'passport-d',
        'passport-s',
        'passport-n',
        'passport-e',
        'passport-code',
        'bank-type',
        'document',
    }
    NONRESIDENT_REQUIRED_FIELDS = {
        'type',
        'lname',
        'fname',
        'birthday',
        'phone',
        'email',
        'city',
        'country',
        'postcode',
        'postaddress',
        'ben-bank',
        'swift',
        'document',
    }
    BANK_TYPE_REQUIRED_FIELDS = {
        '0': [],
        '1': ['person-account', 'bik'],
        '2': ['person-account', 'bik'],
        '3': ['yamoney-wallet'],
        '7': ['payoneer-wallet'],
    }

    RESIDENT_TYPE = 'ph'
    NONRESIDENT_TYPE = 'ytph'

    reporter = models.OneToOneField(Reporter, related_name='new_payment_info', null=False)
    modified_at = models.DateTimeField(auto_now=True)
    balance_simulation_result = models.TextField(null=True, blank=True)
    encrypted_document_name = models.BinaryField(null=True, blank=True)
    encrypted_document_content = models.BinaryField(null=True, blank=True)
    document_uploaded_at = models.DateTimeField(null=True, blank=True)

    @property
    def datasync_values(self):
        if getattr(self, '_datasync_values', None) is None:
            if self.id:
                self._datasync_values = datasync.get_payment_info_entry(self.reporter.uid, self.id)
                if self._datasync_values.get('cipher_key'):
                    self._load_image(self._datasync_values)
            else:
                self._datasync_values = {}
        return self._datasync_values

    def _save_image(self, image, values):
        if not hasattr(self, 'cipher_key'):
            self.cipher_key = Fernet.generate_key()
        image.seek(0)
        image_data = image.read()
        cipher = Fernet(self.cipher_key)
        self.encrypted_document_name = cipher.encrypt(bytes(image.name.encode('utf-8')))
        self.encrypted_document_content = cipher.encrypt(image_data)
        values['cipher_key'] = self.cipher_key

    def _load_image(self, values):
        doc_buffer = BytesIO()
        key = values.pop('cipher_key').encode('utf-8')  # key должен быть набором байт
        self.cipher_key = key
        cipher = Fernet(key)
        decrypted_image = cipher.decrypt(force_bytes(self.encrypted_document_content))
        decrypted_name = cipher.decrypt(force_bytes(self.encrypted_document_name)).decode('utf-8')
        doc_buffer.write(decrypted_image)
        doc_buffer.name = decrypted_name
        doc_buffer.seek(0)
        values['document'] = doc_buffer

    def save(self, *args, **kwargs):
        self._datasync_values = {k: v for (k, v) in self.datasync_values.items() if k in self.DATASYNC_FIELDS}
        super(NewPaymentInfo, self).save(*args, **kwargs)
        self.datasync_values['id'] = unicode(self.id)
        values = self.datasync_values.copy()
        if 'document' in values:
            doc_buffer = values.pop('document')
            self._save_image(doc_buffer, values)
            super(NewPaymentInfo, self).save(update_fields=['encrypted_document_content', 'encrypted_document_name'])
        datasync.update_or_create_payment_info_entry(self.reporter.uid, values)

    def refresh_from_db(self, using=None, fields=None):
        super(NewPaymentInfo, self).refresh_from_db(using=using, fields=fields)
        if hasattr(self, '_datasync_values'):
            del self._datasync_values

    def __str__(self):
        return str(self.reporter)

    def get_absolute_url(self):
        return reverse('user_profile:edit_payment', kwargs={'pk': self.reporter.pk})

    class Meta:
        app_label = 'core'
        verbose_name_plural = 'new payment info profiles'


class Administrator(AbstractUser):

    class Meta:
        app_label = 'core'

    objects = AdministratorManager()


class Document(models.Model):
    document_id = models.CharField(max_length=100)
    payment_information = models.ForeignKey(PaymentInfo)
    created = models.DateTimeField(auto_now_add=True)

    class Meta:
        app_label = 'core'


class ResponsibleQuerySet(models.QuerySet):
    pass


class ResponsibleManager(models.Manager.from_queryset(ResponsibleQuerySet)):
    def supervisors(self):
        return self.filter(role=Responsible.ROLE_SUPERVISOR)

    def infosec_chairmen(self):
        return self.filter(role=Responsible.ROLE_INFOSEC_CHAIRMAN)

    def infosec_members(self):
        return self.filter(role=Responsible.ROLE_INFOSEC_MEMBER)


class Responsible(models.Model):
    #  Этих людей мы раз в месяц первого числа призываем в тикет для сверки сумм выплат
    ROLE_SUPERVISOR = 'supervisor'
    #  Строка "Председатель комиссии" в протоколе для нерезидентов
    ROLE_INFOSEC_CHAIRMAN = 'infosec_chairman'
    #  "Члены комиссии" в протоколе для нерезидентов
    ROLE_INFOSEC_MEMBER = 'infosec_member'
    # Ответственный от сервиса. Призывается, когда есть расхождения.
    ROLE_BUGBOUNTY_RESPONSIBLE = 'bugbounty_responsible'
    ROLE = (
        (ROLE_SUPERVISOR, _('reward.role_supervisor')),
        (ROLE_INFOSEC_CHAIRMAN, _('reward.role_infosec_chairman')),
        (ROLE_INFOSEC_MEMBER, _('reward.role_infosec_member')),
        (ROLE_BUGBOUNTY_RESPONSIBLE, _('reward.role_bugbounty_responsible'))
    )
    staff_username = models.CharField(max_length=255)
    role = models.CharField(max_length=255, choices=ROLE)
    name = models.CharField(max_length=255)
    name_genitive = models.CharField(max_length=255, null=True, blank=True)

    def __str__(self):
        return '{}, {}'.format(self.staff_username, self.role)

    class Meta:
        app_label = 'core'
        verbose_name = 'person responsible for regular reward processing'
        verbose_name_plural = 'people responsible for regular reward processing'

    objects = ResponsibleManager()
