# coding: utf-8
from __future__ import unicode_literals

from functools import wraps
from lxml import etree
import requests
import sys
import traceback
import xmlrpclib

from django.conf import settings

from core.utils.tvm import get_tvm_ticket
from core.models import Reward, NewPaymentInfo


class BadClientDetailsError(Exception):
    # Не ретраится
    pass


class BalancePushError(Exception):
    # Ретраится
    pass


class BalanceInternalError(BalancePushError):
    pass


class BugbountyInternalError(BalancePushError):
    def __init__(self, *args, **kwargs):
        on_error = kwargs.pop('on_error', False)
        if on_error:
            _, exception, tb = sys.exc_info()
            self.original_error = exception
            self.original_traceback = '\n'.join(traceback.format_tb(tb))
        else:
            self.original_error = None
            self.original_traceback = None
        super(BugbountyInternalError, self).__init__(*args, **kwargs)


def catch_balance_errors(func):
    @wraps(func)
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except (BadClientDetailsError, BalancePushError):
            raise
        except Exception:
            raise BugbountyInternalError('Caught exception while pushing to Balance', on_error=True)
    return wrapper


def try_parse_error(fault_string):
    """
    Пытаемся партить fault-ответ от Баланса. Ищем тег "msg", если не находим, то сдаёмся
    """
    root = etree.fromstring(fault_string)
    messages = [elem for elem in root.getchildren() if elem.tag == 'msg']
    if len(messages) != 1:
        return None
    message = messages[0]
    if not message.text:
        return None
    return message.text


class BalanceClient(object):
    CLIENT_DATASYNC_FIELDS = {'email', 'phone'}
    CURRENCY_ACRONYMS = {
        Reward.C_RUR: 'RUB',
        Reward.C_USD: 'USD',
    }
    RUSSIAN_RESIDENT_TYPE = 'ph'

    CREATE_CLIENT_SUCCESS_CODE = 0

    IGNORED_NONRESIDENT_FIELDS = {'country', 'document'}
    IGNORED_RESIDENT_FIELDS = {'legal-fias-guid', 'document'}

    def get_balance_person_fieldname(self, reporter):
        datasync_values = reporter.new_payment_info.datasync_values
        if datasync_values['type'] == self.RUSSIAN_RESIDENT_TYPE:
            return 'balance_person_id'
        else:
            return 'balance_nonresident_person_id'

    def send_data(self, args, methodname):
        xml = xmlrpclib.dumps(args, methodname)
        headers = {
            'Accept-Encoding': 'gzip',
            'User-Agent': 'xmlrpclib.py/1.0.1 (by www.pythonware.com)',
            'X-Ya-Service-Ticket': get_tvm_ticket(settings.BALANCE_TVM_ID),
            'Content-Type': 'text/xml',
            'Content-Length': str(len(xml)),
        }
        response = None
        try:
            response = requests.post(settings.BALANCE_URL, headers=headers, data=xml, timeout=settings.BALANCE_TIMEOUT)
            response.raise_for_status()
        except requests.exceptions.Timeout:
            raise BalanceInternalError('Timeout from Balance')
        except requests.exceptions.RequestException as err:
            error_message = 'Invalid response from Balance'
            if response:
                additional_info = ': status_code={}, content="{}"'.format(response.status_code, response.content)
            else:
                additional_info = ''
            raise BalanceInternalError(error_message + additional_info)
        try:
            parsed_response = xmlrpclib.loads(response.content)
        except xmlrpclib.Fault as err:  # возможно тут могут быть и внутренние ошибки, но we will never know
            parsed_error = try_parse_error(err.faultString)
            if parsed_error is None:
                parsed_error = err.faultString  # fallback на сырой xml
            raise BadClientDetailsError('Got error from Balance: "{}"'.format(parsed_error))
        except Exception:
            raise BalanceInternalError('Could not parse Balance response: content="{}"'.format(response.content))
        return parsed_response

    def send_create_client(self, uid, params):
        return self.send_data((unicode(uid), params), 'Balance.CreateClient')[0]

    def send_create_person(self, uid, params):
        return self.send_data((unicode(uid), params), 'Balance.CreatePerson')[0][0]

    def send_create_offer(self, uid, params):
        return self.send_data((unicode(uid), params), 'Balance.CreateOffer')[0][0]

    def send_get_client_contracts(self, client_id):
        contracts = self.send_data(({
            'ClientID': client_id, 'ContractType': 'SPENDABLE'
        }, ), 'Balance.GetClientContracts')[0][0]
        return [contract for contract in contracts if contract.get('IS_ACTIVE')]

    @catch_balance_errors
    def create_client(self, reporter, return_params=False):
        try:
            datasync_values = reporter.new_payment_info.datasync_values
        except NewPaymentInfo.DoesNotExist:
            raise BadClientDetailsError('User\'s payment info was not submitted')
        except Exception:
            raise BugbountyInternalError('Could not get payment info from Datasync', on_error=True)
        if datasync_values['type'] == self.RUSSIAN_RESIDENT_TYPE:
            city = datasync_values.get('address-city') or datasync_values.get('legal-address-city')
            name = ' '.join((datasync_values['fname'], datasync_values['mname']))
        else:
            city = datasync_values.get('city')
            name = ' '.join((datasync_values['fname'], datasync_values['lname']))
        if city is None:
            raise BadClientDetailsError(
                'address-city or legal-address-city for resident or city for non-resident must be specified'
            )
        params = {
            'IS_AGENCY': '0',  # наш репортер точно не является агентством
            'CITY': city,
            'NAME': name,
        }
        for field in self.CLIENT_DATASYNC_FIELDS:
            params[field.upper()] = datasync_values[field]
        if reporter.balance_client_id is not None:
            params['CLIENT_ID'] = reporter.balance_client_id

        if return_params:
            return params
        response = self.send_create_client(reporter.uid, params)
        if response[0] == self.CREATE_CLIENT_SUCCESS_CODE:
            client_id = unicode(response[-1])
        else:
            # возможно это не внутренняя ошибка, но как обработать – хз
            raise BalanceInternalError('Bad response from Balance: {}'.format(response[1]))

        reporter.balance_client_id = client_id
        reporter.save(update_fields=('balance_client_id',))
        return client_id

    def get_person_params(self, datasync_values, client_id, person_id=None):
        params = {
            'type': datasync_values['type'],
            'client_id': client_id,
            'is_partner': '1'
        }

        if person_id is not None:
            params['person_id'] = person_id
        if params['type'] == NewPaymentInfo.RESIDENT_TYPE:
            datasync_fields = NewPaymentInfo.RESIDENT_REQUIRED_FIELDS - self.IGNORED_RESIDENT_FIELDS
            if datasync_values.get('bank-type') in NewPaymentInfo.BANK_TYPE_REQUIRED_FIELDS:
                datasync_fields.update(NewPaymentInfo.BANK_TYPE_REQUIRED_FIELDS[datasync_values.get('bank-type')])
        elif params['type'] == NewPaymentInfo.NONRESIDENT_TYPE:
            datasync_fields = NewPaymentInfo.NONRESIDENT_REQUIRED_FIELDS - self.IGNORED_NONRESIDENT_FIELDS
        else:
            datasync_fields = {}

        for field in datasync_fields:
            value = datasync_values.get(field)
            if value is not None:
                params[field] = value

        # https://st.yandex-team.ru/BALANCEDUTY-504
        if params['type'] == NewPaymentInfo.NONRESIDENT_TYPE:
            iban = datasync_values.get('account', '')
            account = datasync_values.get('ben-account', '')
            params.update({
                'iban': iban,
                'account': account,
                'ben-account': params.get('ben-bank', ''),
            })

        return params

    @catch_balance_errors
    def create_person(self, reporter):
        if reporter.balance_client_id is None:
            raise BugbountyInternalError('Cannot create person without client')
        datasync_values = reporter.new_payment_info.datasync_values
        person_id_fieldname = self.get_balance_person_fieldname(reporter)
        params = self.get_person_params(
            datasync_values, reporter.balance_client_id, getattr(reporter, person_id_fieldname))
        response = self.send_create_person(reporter.uid, params)
        person_id = unicode(response)
        setattr(reporter, person_id_fieldname, person_id)
        reporter.save(update_fields=(person_id_fieldname,))
        return response

    @catch_balance_errors
    def create_contract(self, reward, return_params=False):
        reporter = reward.reporter
        client_type = reward.reporter.new_payment_info.datasync_values['type']
        reward.payment_currency = Reward.C_RUR if client_type == NewPaymentInfo.RESIDENT_TYPE else Reward.C_USD
        reward.save(update_fields=['payment_currency'])
        person_id = getattr(reporter, self.get_balance_person_fieldname(reporter))
        if reporter.balance_client_id is None or person_id is None:
            raise BugbountyInternalError('Cannot create person without client and reporter')

        params = {
            'client_id': reporter.balance_client_id,
            'person_id': person_id,
            'firm_id': '1',
            'currency': self.CURRENCY_ACRONYMS[reward.payment_currency],
            'nds': '0',
            'services': [settings.BALANCE_BUGBOUNTY_SERVICE_ID],
            'manager_uid': settings.BALANCE_MANAGER_UID,
            'pay_to': '1',
            'signed': '1',
        }

        current_contracts = self.send_get_client_contracts(reporter.balance_client_id)
        for contract in current_contracts:
            if unicode(contract['PERSON_ID']) == person_id:
                contract_id = contract['EXTERNAL_ID']
                if reporter.balance_contract_id != contract_id:
                    reporter.balance_contract_id = contract_id
                    reporter.save(update_fields=['balance_contract_id'])
                return contract_id

        if return_params:
            return params
        response = self.send_create_offer(reporter.uid, params)
        contract_id = response['EXTERNAL_ID']
        reporter.balance_contract_id = contract_id
        reporter.save(update_fields=['balance_contract_id'])
        return contract_id
