# -*- coding: utf-8 -*-
from __future__ import absolute_import
from __future__ import unicode_literals

import logging
from functools import partial

from blackbox import BlackboxResponseError
from django.conf import settings

from core.exceptions import CoreException

__all__ = [
    'external_userinfo_by_login',
    'external_userinfo_by_uid',
    'BlackboxUser',
    'BlackboxUserError',
]

logger = logging.getLogger(__name__)


blackbox_userinfo_helper = partial(
    settings.EXTERNAL_BLACKBOX_INSTANCE.userinfo,
    userip='::1', emails='getdefault',
    attributes=1008, get_public_id='yes'  # karganov@: technical login, normalized and always present
)


def external_userinfo_by_login(login):
    try:
        user = blackbox_userinfo_helper(login, by_login=True)
    except BlackboxResponseError as exc:
        log_unexpected_blackbox_response(exc)
    else:
        return BlackboxUser(**user)


def external_userinfo_by_uid(uid):
    try:
        user = blackbox_userinfo_helper(uid)
    except BlackboxResponseError as exc:
        log_unexpected_blackbox_response(exc)
    else:
        return BlackboxUser(**user)


def log_unexpected_blackbox_response(exc):
    logger.critical('Unexpected blackbox response: {0.status} {0.content}'.format(exc))


class BlackboxUserError(CoreException):
    pass


class BlackboxUser(object):

    def __init__(self, **kwargs):
        self._kwargs = kwargs
        fields = kwargs.get('fields', dict())
        aliases = kwargs.get('fields', dict()).get('aliases', list())
        alias = filter(lambda x: x[0] == '1', aliases) if aliases else ''
        self.username = alias[0][1] if alias else ''
        self.uid = kwargs.get('uid', '')
        self.email = kwargs.get('emails')
        self.avatar = fields.get('default_avatar_id')
        self.public_id = kwargs.get('public_id')
        self.display_name = fields.get('display_name')

    @property
    def username(self):
        return self._username

    @username.setter
    def username(self, value):
        self._username = value

    @property
    def uid(self):
        return self._id

    @uid.setter
    def uid(self, value):
        try:
            self._id = int(value)
        except ValueError:
            raise BlackboxUserError('Not found')

    @property
    def email(self):
        return self._email

    @email.setter
    def email(self, address_list):
        if address_list:
            self._email = address_list[0].get('address')
        else:
            self._email = None
            logger.critical('User %s has no email: %s', self.username, self._kwargs)

    @property
    def avatar(self):
        return self._avatar

    @avatar.setter
    def avatar(self, value):
        self._avatar = value

    @property
    def public_id(self):
        return self._public_id

    @public_id.setter
    def public_id(self, value):
        self._public_id = value

    @property
    def display_name(self):
        return self._display_name

    @display_name.setter
    def display_name(self, value):
        self._display_name = value
