# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import datetime
import functools

import pytz

from django.conf import settings


__all__ = [
    'get_current_month',
    'get_previous_month',
    'get_previous_month_range',
    'today', 'yesterday',
]


def localize(func):
    local_timezone = pytz.timezone(settings.TIME_ZONE)

    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        return local_timezone.localize(func(*args, **kwargs))

    return wrapper


def month_first_day(func):
    """Decorated function result is truncated to first day of the month."""
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        return func(*args, **kwargs).replace(
            day=1, hour=0, minute=0, second=0, microsecond=0)
    return wrapper


@month_first_day
def get_previous_month():
    """Get previous month as timezone aware datetime object."""
    month_first_day = get_current_month()
    return month_first_day - datetime.timedelta(days=1)


@month_first_day
def get_current_month():
    """Get current month as timezone aware datetime object."""
    return today()


def get_previous_month_range():
    """Return tuple of datetimes bounding previous month."""
    return (get_previous_month(), get_current_month())


def get_range_for_last(days):
    last = today()
    first = last - datetime.timedelta(days=days)
    return first, last


today = localize(datetime.datetime.today)


def yesterday():
    return today() - datetime.timedelta(days=1)


# TODO(remedy) This data migration is still pending.
def initialize_month(apps, schema_editor):
    """Create datettime.date object with day=1.

    This function is used in HallOfFameGroup migration.
    """
    HallOfFameGroup = apps.get_model('core', 'HallOfFameGroup')
    for group in HallOfFameGroup.objects.all():
        group.group_month = datetime.date(
            int(group.year), int(group.month), 1)
        group.save()
