# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

from django.conf import settings
from django.core import mail

from core import exceptions
from core import models


logger = logging.getLogger(__name__)


class BadgeMessageError(exceptions.ModelException):
    pass


class BadgeMessage(object):

    def __init__(self, user_badge, deprive=False):
        super(BadgeMessage, self).__init__()
        self._deprive = deprive
        self._user_badge = user_badge
        self._user = user_badge.user
        self._badge = user_badge.badge
        self._message = self._init_message()

    def _init_message(self):
        mail_template = self.get_mail_template()
        badge_name = self.get_localized_badge_name()
        body = mail_template.template.format(
            badge=badge_name, username=self._user.login)
        return mail.EmailMessage(
            subject=mail_template.subject.format(badge=badge_name),
            from_email=settings.FROM_EMAIL,
            to=[self._user.default_email],
            body=body)

    def get_mail_template(self):
        code=self.get_template_code()
        language=self.get_language()
        try:
            template = models.MailTemplate.objects.get(
                code=code, language=language)
        except models.MailTemplate.DoesNotExist:
            raise BadgeMessageError(
                'Template %s(%s) does not exist' % (code, language))
        else:
            return template

    def get_template_code(self):
        if self._deprive:
            return 'badge_deprive'
        else:
            return 'badge'

    def get_localized_badge_name(self):
        if self.get_language() == models.MailTemplate.LANGUAGE_RUSSIAN:
            return self._badge.name
        else:
            return self._badge.name_en

    def get_language(self):
        if self._user.is_russian_resident:
            return models.MailTemplate.LANGUAGE_RUSSIAN
        else:
            return models.MailTemplate.LANGUAGE_ENGLISH

    def send(self):
        self._message.send()
        logger.info('Message from %s to %s was sent',
                    self._message.from_email,
                    ', '.join(self._message.to))
